﻿// System.Security.Cryptography.Mugi.cs
// 1.0.*.1

namespace System.Security.Cryptography
{
    /// <summary>
    /// MUGI 擬似乱数ジェネレータ (http://www.hitachi.co.jp/rd/yrl/crypto/mugi/) を実装します。このクラスは継承できません。
    /// </summary>
    /// <remarks>この実装は http://www.ipa.go.jp/security/enc/CRYPTREC/fy15/doc/10_02jspec.pdf に基づいています。</remarks>
    sealed partial class Mugi : RandomNumberGenerator
    {
        private Hitachi.Software.Cryptography.Mugi _Mugi;
        private Byte[] _Cache;
        private Int32 _CacheIndex;
        /// <summary>
        /// 指定した、128 ビット (16 バイト) の キーおよび初期化ベクタを使用して、<see cref="T:System.Security.Cryptography.Mugi"/> クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="key">キーを格納したバイト配列。</param>
        /// <param name="iv">初期化ベクタを格納したバイト配列。</param>
        /// <exception cref="T:System.ArgumentNullException"><paramref name="key"/> または <paramref name="iv"/> が null です。</exception>
        /// <exception cref="T:System.Security.Cryptography.CryptographicException"><paramref name="key"/> または <paramref name="iv"/> の要素数が 16 未満です。</exception>
        public Mugi(Byte[] key, Byte[] iv)
        {
            _Mugi = new Hitachi.Software.Cryptography.Mugi(key, iv);
        }
        /// <summary>
        /// バイト配列に、暗号化に使用するランダムな値の厳密なシーケンスを設定します。
        /// </summary>
        /// <param name="data">暗号化に使用するランダムな値の厳密なシーケンスを格納する配列。</param>
        /// <exception cref="T:System.ArgumentNullException"><paramref name="data"/> が null です。</exception>
        public override void GetBytes(Byte[] data)
        {
            if(data == null)
                throw new ArgumentNullException("data", EnvironmentExtension.GetResourceString("ArgumentNull_Array"));
            Byte[] c = _Cache;
            Int32 ci = _CacheIndex;
            try
            {
                Int32 r;
                for(Int32 i = 0; (r = data.Length - i) != 0; )
                {
                    if(c == null || ci >= c.Length)
                    {
                        c = _Mugi.NextRound(8);
                        ci = 0;
                    }
                    Int32 j = Math.Min(r, c.Length - ci);
                    do
                        data[i++] = c[ci++];
                    while(--j != 0);
                }
            }
            finally
            {
                _Cache = c;
                _CacheIndex = ci;
            }
        }
        /// <summary>
        /// バイト配列に、暗号化に使用するランダムな 0 以外の値の厳密なシーケンスを設定します。
        /// </summary>
        /// <param name="data">暗号化に使用するランダムな 0 以外の値の厳密なシーケンスを格納する配列。</param>
        /// <exception cref="T:System.ArgumentNullException"><paramref name="data"/> が null です。</exception>
        public override void GetNonZeroBytes(Byte[] data)
        {
            if(data == null)
                throw new ArgumentNullException("data", EnvironmentExtension.GetResourceString("ArgumentNull_Array"));
            Byte[] c = _Cache;
            Int32 ci = _CacheIndex;
            try
            {
                Int32 r;
                for(Int32 i = 0; (r = data.Length - i) != 0; )
                {
                    if(c == null || ci >= c.Length)
                    {
                        c = _Mugi.NextRound(8);
                        ci = 0;
                    }
                    Int32 j = Math.Min(r, c.Length - ci);
                    do
                    {
                        Byte b;
                        if((b = c[ci++]) != 0)
                            data[i++] = b;
                    }
                    while(--j != 0);
                }
            }
            finally
            {
                _Cache = c;
                _CacheIndex = ci;
            }
        }
    }
}
