using System;
using System.Collections.Generic;
using System.Text;

namespace Chemics
{
    struct SphericalHarmonicsParameters
    {
        //angular momentum
        public int AzimuthalQuantumNumber;
        //projection of angular momentum
        public int MagneticQuantumNumber;

        public override string ToString()
        {
            return string.Format(
                "AngularMomentum={0}\nMagneticQuantumNumber={1}",
                AzimuthalQuantumNumber,
                MagneticQuantumNumber
                );
        }
    }

    class SphericalHarmonics
    {
        public SphericalHarmonicsParameters Parameters;

        public Complex GetValue(double theta, double phi)
        {
            switch (Parameters.AzimuthalQuantumNumber)
            {
                case 0: return GetValueL0(theta, phi);
                case 1: return GetValueL1(theta, phi);
                case 2: return GetValueL2(theta, phi);
                case 3: return GetValueL3(theta, phi);
                case 4: return GetValueL4(theta, phi);
                default: return 0;
            }
        }

        public double GetValueL0(double angleFromZ, double angleZ)
        {
            return 1 / Math.Sqrt(4 * Math.PI);
        }

        public Complex GetValueL1(double theta, double phi)
        {
            switch (Parameters.MagneticQuantumNumber)
            {
                case 0: return Math.Sqrt(3 / Math.PI) / 2 * Math.Cos(theta);
                case -1: return Math.Sqrt(3 / (Math.PI * 2)) / 2 * exp(-phi) * Math.Sin(theta);
                case 1: return -Math.Sqrt(3 / (Math.PI * 2)) / 2 * exp(phi) * Math.Sin(theta);
                default: return 0;
            }
        }

        public Complex GetValueL2(double theta, double phi)
        {
            double sin = Math.Sin(theta);
            double cos = Math.Cos(theta);

            switch (Parameters.MagneticQuantumNumber)
            {
                case -2: return 1 / 4.0 * Math.Sqrt(15 / (Math.PI * 2)) * exp(-2 * phi) * sin * sin;
                case -1: return 1 / 2.0 * Math.Sqrt(15 / (2 * Math.PI)) * exp(-phi) * Math.Sin(theta) * Math.Cos(theta);
                case 0: return 1 / 4.0 * Math.Sqrt(5 / Math.PI) * (3 * cos * cos - 1);
                case 1: return -1 / 2.0 * Math.Sqrt(15 / (2 * Math.PI)) * exp(phi) * Math.Sin(theta) * Math.Cos(theta);
                case 2: return 1 / 4.0 * Math.Sqrt(15 / (2 * Math.PI)) * exp(2 * phi) * Math.Sin(theta) * Math.Sin(theta);
                default: return 0;
            }
        }

        public Complex GetValueL3(double theta, double phi)
        {
            double sin = Math.Sin(theta);
            double cos = Math.Cos(theta);

            switch (Parameters.MagneticQuantumNumber)
            {
                case -3: return 1 / 8.0 * Math.Sqrt(35 / Math.PI) * exp(-3 * phi) * sin * sin * sin;
                case 3: return -1 / 8.0 * Math.Sqrt(35 / Math.PI) * exp(3 * phi) * sin * sin * sin;

                case -2: return 1 / 4.0 * Math.Sqrt(105 / (2 * Math.PI)) * exp(-2 * phi) * sin * sin * cos;
                case 2: return 1 / 4.0 * Math.Sqrt(105 / (2 * Math.PI)) * exp(2 * phi) * sin * sin * cos;

                case -1: return 1 / 8.0 * Math.Sqrt(21 / Math.PI) * exp(-phi) * sin *(5 * cos * cos - 1);
                case 1: return -1 / 8.0 * Math.Sqrt(21 / Math.PI) * exp(phi) * sin * (5 * cos * cos - 1);

                case 0: return 1 / 4.0 * Math.Sqrt(7 / Math.PI) * (5 * cos * cos * cos - 3 * cos);
                default: return 0;
            }
        }

        public Complex GetValueL4(double theta, double phi)
        {
            double sin = Math.Sin(theta);
            double cos = Math.Cos(theta);

            switch (Parameters.MagneticQuantumNumber)
            {
                case -4: return 3 / 16.0 * Math.Sqrt(35 / (2 * Math.PI)) * exp(-4 * phi) * sin * sin * sin * sin;
                case 4: return 3 / 16.0 * Math.Sqrt(35 / (2 * Math.PI)) * exp(4 * phi) * sin * sin * sin * sin;

                case -3: return 3 / 8.0 * Math.Sqrt(35 / Math.PI) * exp(-3 * phi) * sin * sin * sin * cos;
                case 3: return -3 / 8.0 * Math.Sqrt(35 / Math.PI) * exp(3 * phi) * sin * sin * sin * cos;

                case -2: return 3 / 8.0 * Math.Sqrt(5 / (2 * Math.PI)) * exp(-2 * phi) * sin * sin * (7 * cos * cos - 1);
                case 2: return 3 / 8.0 * Math.Sqrt(5 / (2 * Math.PI)) * exp(2 * phi) * sin * sin * (7 * cos * cos - 1);

                case -1: return 3 / 8.0 * Math.Sqrt(5 / Math.PI) * exp(-phi) * sin * (7 * cos * cos * cos - 3 * cos);
                case 1: return -3 / 8.0 * Math.Sqrt(5 / Math.PI) * exp(phi) * sin * (7 * cos * cos * cos - 3 * cos);

                case 0: return 3 / 16.0 * Math.Sqrt(1 / Math.PI) * (35 * cos * cos * cos * cos - 30 * cos * cos + 3);
                default: return 0;
            }
        }

        Complex exp(double imaginaryNumber)
        {
            return new Complex(
                Math.Cos(imaginaryNumber),
                Math.Sin(imaginaryNumber)
                );
        }
    }

    struct Complex
    {
        public double R;
        public double I;

        public Complex(double realNumber, double imaginaryNumber)
        {
            this.R = realNumber;
            this.I = imaginaryNumber;
        }

        public double LengthSquared()
        {
            return R * R + I * I;
        }

        public static Complex operator + (Complex c, double r){
            c.R += r;
            return c;
        }

        public static Complex operator *(Complex c1, Complex c2)
        {
            Complex result = new Complex();
            result.R = c1.R * c2.R - c1.I * c2.I;
            result.I = c1.R * c2.I + c1.I * c2.R;
            return result;
        }

        public static implicit operator Complex(double real)
        {
            return new Complex(real, 0);
        }
    }
}
