/****************************************************************************

	HoneyRadio
		Copyright(C) 2012 Mr.Honey

****************************************************************************/

#include "Main.h"
#include "Display.h"
#include "Volume.h"
#include "FMRadio.h"

//=================================================================

// RDA5800C WX^{l

// REG 02H
// B15: Audio Output High-Z Disable = High impedance
// B14: Mute Disable = Mute
// B13: Mono Select = Stereo
// B12: Bass Boost = Disabled
// B11: 1 
//  B9: Seek Up = Seek down
//  B8: Seek = Disable
//  B7: 1 
//  B0: Power Up Enable = Disabled
const static WORD s_nBase02 = 0x0880;

// REG 04H
// B14: Seek/Tune Complete Interrupt Enable = Disable Interrupt
// B11: De-emphasis = 75us
//  B6: I2S Bus Enable = disabled
// B5:4: GPIO 3 = High impedance
// B3:2: GPIO 2 = High impedance
// B1:0: GPIO 1 = High impedance
const static WORD s_nBase04 = 0x0000;

// REG 05H
// B15: INT Mode Select = Generate 5ms interrupt
// B13:8: Seek Threshold in Logarithmic = 010000
// B7:4: DSP Volume Control = max 0db
// B3:0: DAC Gain Control Bits (Volume) = max
const static WORD s_nBase05 = 0x10FF;

//=================================================================

#define ST_NONE			0
#define ST_WAIT			1
#define ST_PLAY			2
#define ST_SEEK			3

#define MODE_MANU		0
#define MODE_PSET		1
#define MAX_PSET		16

//=================================================================

static void Start(void);
static void Stop(void);
static void ChanUpDn(BOOL fUp);
static void FreqUpDn(BOOL fUp);
static void Seek(BOOL fUp);
static void SetFreq(WORD freq);
static WORD GetFreq(void);
static void Write(BYTE addr, WORD data);
static WORD Read(BYTE addr);
static void LoadPsets(void);
static void Update(void);
static const char* FindName(void);

//=================================================================

static struct _preset {
	char name[13];
	WORD freq;
} s_Psets[MAX_PSET];
static int s_nPset = 0;

static WORD s_nReg02;
static WORD s_nReg04;
static WORD s_nReg05;

static int s_nState = ST_NONE;
static int s_nMode;
static int s_nChan;
static WORD s_wFreq;
static DWORD s_dwTick;

//=================================================================

// 
extern void InitFMR(void)
{
	// |[g
	FM_SEN_IO = 1;
	FM_SEN_TRIS = OUTPUT;
	FM_SDIO_IO = 0;
	FM_SDIO_TRIS = OUTPUT;
	FM_SCLK_IO = 0;
	FM_SCLK_TRIS = OUTPUT;

	// WX^
	Write(0x02, s_nReg02 = s_nBase02);
	Write(0x04, s_nReg04 = s_nBase04);
	Write(0x05, s_nReg05 = s_nBase05);
}

// Jn
extern void StartFMR(void)
{
	ASSERT(s_nState == ST_NONE);

	// ݒ蕜
	s_nMode = g_Data.nLastFmMode;
	s_nChan = g_Data.nLastFmChan;
	s_wFreq = g_Data.nLastFmFreq;
	if (s_wFreq == 0) {
		s_wFreq = 7600;
	}
	// xJn
	s_dwTick = TickGet() + TICKS_OFMS(500);
	s_nState = ST_WAIT;
	Update();
}

// ~
extern void StopFMR(void)
{
	ASSERT(s_nState != ST_NONE);

	// ~
	if (s_nState >= ST_PLAY)
	{
		SetDefClock();
		SetMute(TRUE);
		Stop();
	}
	// ݒۑ
	g_Data.nLastFmMode = (BYTE)s_nMode;
	g_Data.nLastFmChan = (BYTE)s_nChan;
	g_Data.nLastFmFreq = s_wFreq;

	// ~
	s_nState = ST_NONE;
}

// ^XN
extern void TaskFMR(void)
{
	switch (s_nState)
	{
	case ST_WAIT:
		ASSERT(GetSystemClock() == DEFAULT_CLK);
		if (TickGet() >= s_dwTick)
		{
			// xJn
			if (s_nPset == 0)
			{
				// vZbg[h
				LoadPsets();
				Update();
			}
			// MJn
			Start();
			SetMute(FALSE);

			// NbN_E
			SetClock(CLK_32KHz);
			s_nState = ST_PLAY;
		}
		break;

	case ST_PLAY:
		if (s_nMode == MODE_MANU)
		{
			// }jA[h
			int nSW = GetPressedSW(SW_PREV | SW_NEXT);
			if (nSW == SW_NONE)
			{
				nSW = GetHoldingSW(SW_PREV | SW_NEXT);
				if (nSW != SW_NONE)
				{
					// F}jA[hֈڍsăV[N
					s_nState = ST_SEEK;
					Update();
					Seek(nSW == SW_NEXT);
					s_nState = ST_PLAY;
					Update();
					ResetInput();
				}
				else if (GetPressedSW(SW_FUNC))
				{
					if (IsDisplay())
					{
						// [hؑ
						s_nMode = MODE_PSET;
						Update();
					}
					else {
						DisplayCtrl(DISPLAY_AUTO);
					}
				}
				else if (GetHoldingSW(SW_FUNC))
				{
					// It^C}[
					SetOffTimer(TRUE);
					ResetInput();
				}
			}
			else
			{
				if (IsDisplay())
				{
					// FgUp/Down
					FreqUpDn(nSW == SW_NEXT);
					Update();
				}
				else {
					DisplayCtrl(DISPLAY_AUTO);
				}
			}
		}
		else
		{
			// vZbg[h
			int nSW = GetPressedSW(SW_PREV | SW_NEXT);
			if (nSW == SW_NONE)
			{
				nSW = GetHoldingSW(SW_PREV | SW_NEXT);
				if (nSW != SW_NONE)
				{
					// F}jA[hֈڍsăV[N
					s_nMode = MODE_MANU;
					s_nState = ST_SEEK;
					Update();
					Seek(nSW == SW_NEXT);
					s_nState = ST_PLAY;
					Update();
					ResetInput();
				}
				else if (GetPressedSW(SW_FUNC))
				{
					if (IsDisplay())
					{
						// [hؑ
						s_nMode = MODE_MANU;
						Update();
					}
					else {
						DisplayCtrl(DISPLAY_AUTO);
					}
				}
				else if (GetHoldingSW(SW_FUNC))
				{
					// It^C}[
					SetOffTimer(TRUE);
					ResetInput();
				}
			}
			else
			{
				if (IsDisplay())
				{
					// F`lUp/Down
					ChanUpDn(nSW == SW_NEXT);
					Update();
				}
				else {
					DisplayCtrl(DISPLAY_AUTO);
				}
			}
		}
		break;

	default:
		ASSERT(s_nState == ST_NONE);
	}
}

//=================================================================

// Jn
static void Start(void)
{
	// B0: Power Up Enable -> Enable
	Write(0x02, s_nReg02 |= 0x0001);
	DelayMS(300); // Iǂ̎sh~

	// I
	SetFreq(s_wFreq);

	// B15: Audio Output High-Z Disable -> Normal operation
	Write(0x02, s_nReg02 |= 0x8000);
	DelayMS(50); // |bvh~

	// B14: Mute Disable -> Normal operation
	Write(0x02, s_nReg02 |= 0x4000);
}

// ~
static void Stop(void)
{
	// B14: Mute Disable -> Mute
	Write(0x02, s_nReg02 &= ~0x4000);

	// B15: Audio Output High-Z Disable -> High impedance
	Write(0x02, s_nReg02 &= ~0x8000);

	// B0: Power Up Enable -> Disabled
	Write(0x02, s_nReg02 &= ~0x0001);
}

// I(`l)
static void ChanUpDn(BOOL fUp)
{
	if (s_nPset > 0)
	{
		ASSERT(0 <= s_nChan && s_nChan < s_nPset);

		// ̃vZbgԍ
		if (s_wFreq != s_Psets[s_nChan].freq)
		{
			// ݂̎gɋ߂`l
			if (fUp)
			{
				for (s_nChan = 0; s_nChan < s_nPset; s_nChan++)
				{
					if (s_Psets[s_nChan].freq > s_wFreq)
						break;
				}
			}
			else
			{
				for (s_nChan = s_nPset - 1; s_nChan >= 0; s_nChan--)
				{
					if (s_Psets[s_nChan].freq < s_wFreq)
						break;
				}
			}
		}
		else {
			s_nChan += ((fUp)? 1 : -1);
		}

		// ɖ߂
		if (s_nChan < 0) {
			s_nChan = s_nPset - 1;
		}
		else if (s_nChan >= s_nPset) {
			s_nChan = 0;
		}

		// I
		SetFreq(s_Psets[s_nChan].freq);
	}
}

// I(g)
static void FreqUpDn(BOOL fUp)
{
	ASSERT(7600 <= s_wFreq && s_wFreq <= 10800);

	if (fUp)
	{
		if (s_wFreq < 10800) {
			SetFreq(s_wFreq += 10);
		}
		else {
			SetFreq(s_wFreq = 7600);
		}
	}
	else
	{
		if (s_wFreq > 7600) {
			SetFreq(s_wFreq -= 10);
		}
		else {
			SetFreq(s_wFreq = 10800);
		}
	}
}

// V[N
static void Seek(BOOL fUp)
{
	// B14: Mute Disable -> Mute
	Write(0x02, s_nReg02 &= ~0x4000);

	// B9: Seek Up
	// B8: Seek = Enable
	Write(0x02, s_nReg02 | ((fUp)? 0x0300 : 0x0100));

	// B8: Seek
	do {
		DelayMS(10);
		ClearWDT();
	}
	while ((Read(0x02) & 0x0100) != 0);

	// g擾
	SetFreq(GetFreq()); // SetFreq()ȂƂȂ̃V[NoȂ

	// B14: Mute Disable -> Normal operation
	Write(0x02, s_nReg02 |= 0x4000);
}

// gݒ肷
static void SetFreq(WORD freq)
{
	ASSERT(7600 <= freq && freq <= 10800);
	s_wFreq = freq;

	// B14: Mute Disable -> Mute
	Write(0x02, s_nReg02 &= ~0x4000);

	// REG 03H
	// B15:8: Channel Select
	// B2: Channel Spacing = see SPACE
	// B1: Band Select
	// B0: Channel Spacing = 100 kHz
	BYTE n;
	if (freq <= 9100) {
		// B15:8: Channel Select -> freq - 7600
		// B1: Band Select = 76-91 MHz (Japan)
		freq = (freq - 7600) / 10;
		n = 0x02;
	}
	else {
		// B15:8: Channel Select -> freq - 8750
		// B1: Band Select -> 87.5-108 MHz (US/Europe)
		freq = (freq - 8750) / 10;
		n = 0x00;
	}

	Write(0x03, MAKEWORD(freq, n));
	DelayMS(100); // ԑ҂肷

	// B14: Mute Disable -> Normal operation
	Write(0x02, s_nReg02 |= 0x4000);
}

// g擾
static WORD GetFreq(void)
{
	// REG 03H
	// B15:8: Channel Select
	// B2: Channel Spacing
	// B1: Band Select
	// B0: Channel Spacing
	WORD reg = Read(0x03);
	BYTE sep = (reg & 0x0004)? 5 : ((reg & 0x0001)? 20 : 10);
	return HIBYTE(reg) * sep + ((reg & 0x0002)? 7600 : 8750);
}

// M
static void Write(BYTE addr, WORD data)
{
	WORD bits, mask;

	FM_SEN_IO = 0;

	// Address (A8-A5,0,A3-A0)
	bits = ((WORD)addr << 1) & 0x1E0;
	bits |= addr & 0x0F;

	mask = 0x0100; // 9Bits
	do {
		FM_SDIO_IO = (bits & mask)? 1 : 0;
		DelayUS(10);
		FM_SCLK_IO = 1;
		DelayUS(10);
		FM_SCLK_IO = 0;
	}
	while (mask >>= 1);

	// Data
	mask = 0x8000; // 16Bits
	do {
		FM_SDIO_IO = (data & mask)? 1 : 0;
		DelayUS(10);
		FM_SCLK_IO = 1;
		DelayUS(10);
		FM_SCLK_IO = 0;
	}
	while (mask >>= 1);

	FM_SDIO_IO = 0;
	FM_SEN_IO = 1;
}

// M
static WORD Read(BYTE addr)
{
	WORD bits, mask, data = 0;

	FM_SEN_IO = 0;

	// Address (A8-A5,1,A3-A0)
	bits = ((WORD)addr << 1) & 0x1E0;
	bits |= 0x10 | (addr & 0x0F);

	mask = 0x0100; // 9Bits
	do {
		FM_SDIO_IO = (bits & mask)? 1 : 0;
		DelayUS(10);
		FM_SCLK_IO = 1;
		DelayUS(10);
		FM_SCLK_IO = 0;
	}
	while (mask >>= 1);

	FM_SDIO_IO = 0;
	FM_SDIO_TRIS = INPUT;

	// Data
	mask = 0x8000; // 16Bits
	do {
		FM_SCLK_IO = 1;
		DelayUS(10);
		data = (data << 1) | FM_SDIO_IO;
 		FM_SCLK_IO = 0;
		DelayUS(10);
	}
	while (mask >>= 1);

	FM_SDIO_TRIS = OUTPUT;
	FM_SEN_IO = 1;

	return data;
}

//=================================================================

// vZbg[h
static void LoadPsets(void)
{
	setting_t Setting;

	// ݒt@CI[v
	if (!OpenSettings(&Setting))
	{
	    LOG("Settings error");
		return;
	}

	// ݒǂݏo
	for (s_nPset = 0; s_nPset < MAX_PSET; s_nPset++)
	{
		char keyF[10];
		sprintf(keyF, "FM%02d-Freq", s_nPset + 1);
		if (ReadSettingNum(&Setting, keyF))
		{
			char keyN[10];
			sprintf(keyN, "FM%02d-Name", s_nPset + 1);
			if (ReadSettingStr(&Setting, keyN))
			{
				ASSERT(strlen(Setting.Str) < sizeof(s_Psets[s_nPset].name));
				strcpy(s_Psets[s_nPset].name, Setting.Str);
				s_Psets[s_nPset].freq = (WORD)Setting.Num;
				continue;
			}
		}
		break;
	}
}

// \XV
static void Update(void)
{
	PushClock(CLK_8MHz);

	if (s_nState != ST_SEEK)
	{
		char buf[32];
		sprintf(buf, "FM:%3d.%dMHz", s_wFreq / 100, (s_wFreq % 100) / 10);
		DisplayText(0, 0, buf);

		if (s_nMode == MODE_MANU)
		{
			/* FM:000.0MHz[   ] */
			/* --- NNNNNNNNNNNN */
			sprintf(buf, "--- %s             ", FindName());
			buf[16] = '\0';
			DisplayText(0, 1, buf);
		}
		else
		{
			ASSERT(s_nMode == MODE_PSET);

			/* FM:000.0MHz[   ] */
			/* P00 NNNNNNNNNNNN */
			if (s_wFreq == s_Psets[s_nChan].freq)
			{	
				sprintf(buf, "P%02d %s             ", s_nChan + 1, s_Psets[s_nChan].name);
				buf[16] = '\0';
				DisplayText(0, 1, buf);
			}
			else
			{
				sprintf(buf, "P-- %s             ", FindName());
				buf[16] = '\0';
				DisplayText(0, 1, buf);
			}
		}
	}
	else
	{
		/* FM:---.-MHz[   ] */
		/*  Seeking...      */
		DisplayText(0, 0, "FM:---.-MHz");
		DisplayText(0, 1, " Seeking...     ");
	}

	PopClock();

	DisplayCtrl(DISPLAY_AUTO);
}

// ݂̕ǖ
static const char* FindName(void)
{
	int i;
	for (i = 0; i < s_nPset; i++)
	{
		if (s_Psets[i].freq == s_wFreq)
			return s_Psets[i].name;
	}
	return "------------";
}
