#include <stddef.h>
#include "3048s.h"
#include "h8.h"
#include "baselib.h"
#include "/cygdrive/c/projects/picwriter_2005/WProtocol.h"


#undef VERBOSE
#undef LINETEST
#undef USEMESSAGE
#undef SLOWSERIAL

typedef unsigned char	byte;
typedef unsigned short	word;

void s_putc(char c);
void s_puts(const char* str);
void wait_count(unsigned long n);

#define	H8_CLOCK		16
#define	WAIT_LOOP_CYCLE	4		// wait_count do 4 clock loops
#define	WAIT_CLOCKX			(H8_CLOCK / WAIT_LOOP_CYCLE)

#define	WAIT_MICROSEC(n)	wait_count(WAIT_CLOCKX * (n))
#define	WAIT_MILLISEC(n)	wait_count(WAIT_CLOCKX * (n) * 1000)

#define	WRITER		PB

#ifdef USEMESSAGE
#define	MESSAGE(text)	stxetx_message(text)
#else
#define	MESSAGE(text)	while (0)
#endif

#define	CVPP		7
#define	CVDD		6
#define	OPT4		5
#define	OPT3		4
#define	OPT2		3
#define	OPT1		2
#define	MCLR		OPT2
#define	PGM			OPT1
#define	CLOCK		1
#define	DATA		0

#define	COMMAND_LOAD_CONFIGURATION	0x00
#define	COMMAND_INCREMENT_ADDRESS						0x06
#define	COMMAND_READ_DATA_FROM_PROGRAM_MEMORY			0x04

#define	TSET0		100
#define	THLD0		5000
#define	TSET1		100
#define	THLD1		100
#define	TDLY1		1000
#define	TDLY2		1000
#define	TDLY3		80

#ifdef SLOWSERIAL
long	linewait = 10000;
long	syncwait = 10000;
#else
long	linewait = 0;
long	syncwait = 0;
#endif

void binprint(byte n)
{
	int i;

	for (i = 0; i < 8; i++) {
		if(n & 0x80)
			s_putc('1');
		else
			s_putc('0');
		n <<= 1;
	}
}

void showline()
{
	binprint(WRITER.DR.BYTE);
}

void printhex4(byte n)
{
	const char* str = "0123456789ABCDEF";
	n &= 0x0f;
	s_putc(str[n]);
}

void printhex8(byte n)
{
	printhex4(n >> 4);
	printhex4(n);
}

void printhex16(word n)
{
	printhex8(n >> 8);
	printhex8(n);
}

int s_getc()
{
	int	r;

	while (!SCI1.SSR.BIT.RDRF);
	r = SCI1.RDR & 0xff;
	SCI1.SSR.BIT.RDRF = 0;
	return r;
}

void waitkey(const char* msg)
{
	s_puts(msg);
	s_getc();
	s_puts("\r\n");
}

void wait(word nsec)
{
	nsec /= 400;
	while (nsec-- != 0);
}

void bitop(int bit, int fset)
{
	byte	f;

#ifdef VERBOSE
	showline();
	s_puts("  ");
	printhex8(WRITER.DR.BYTE);
	s_puts(" bit ");
	printhex4(bit);
	s_puts("=");
	printhex4(fset);
#endif

	f = 1 << bit;
	if (fset) 
		WRITER.DR.BYTE |= f;
	else
		WRITER.DR.BYTE &= ~f;
#ifdef VERBOSE
	s_puts("\r\n");
#endif
}

int bitget(int bit)
{
	byte	f;

	f = 1 << bit;
	return WRITER.DR.BYTE & f;
}

void outline(int data)
{
	WRITER.DR.BYTE = data;
}

void s_putc(char c)
{
	while (!SCI1.SSR.BIT.TDRE);
	SCI1.TDR = c;
	while (!SCI1.SSR.BIT.TDRE);
	SCI1.SSR.BIT.TDRE = 0;
}

void s_puts(const char* str)
{
	while (*str) {
		s_putc(*str++);
	}
}

void stxetx_message(const char* str)
{
	s_putc(STX);
	s_puts(str);
	s_putc(ETX);
}

void write_command6(byte cmd)
{
	int	i;

	WRITER.DDR = 0xff;
	for (i = 0; i < 6; i++) {
		bitop(CLOCK, 1);
		bitop(DATA, cmd & 1);
		wait(TSET1);
		bitop(CLOCK, 0);
		wait(THLD1);
		cmd >>= 1;
	}
}

void write_data14(word data)
{
	int	i;

	WRITER.DDR = 0xff;
	data &= 0x3fff;
	data <<= 1;
	for (i = 0; i < 16; i++) {
		bitop(CLOCK, 1);
		bitop(DATA, data & 1);
		wait(TSET1);
		bitop(CLOCK, 0);
		wait(THLD1);
		data >>= 1;
	}
}

word read_data14()
{
	word data;
	int	i;
	int	r;

	WRITER.DDR = 0xfe;
	data = 0;
	for (i = 0; i < 16; i++) {
		data >>= 1;
		bitop(CLOCK, 1);
		wait(TDLY3);
		r = bitget(DATA);
		data |= r ? 0x8000 : 0x0000;
		bitop(CLOCK, 0);
	}
	data >>= 1;
	data &= 0x3fff;
	return data;
}

#ifdef LINETEST
void linetest()
{
waitkey("hit any to CVPP on\r\n");
	bitop(CVPP, 1);
waitkey("hit any to CVDD on\r\n");
	bitop(CVDD, 1);
waitkey("hit any to OPT4 on\r\n");
	bitop(OPT4, 1);
waitkey("hit any to OPT3 on\r\n");
	bitop(OPT3, 1);
waitkey("hit any to OPT2 on\r\n");
	bitop(OPT2, 1);
waitkey("hit any to OPT1 on\r\n");
	bitop(OPT1, 1);
waitkey("hit any to CLOCK on\r\n");
	bitop(CLOCK, 1);
waitkey("hit any to DATA on\r\n");
	bitop(DATA, 1);
waitkey("hit any to ALL off\r\n");
	WRITER.DR.BYTE = 0x00;
}
#endif

byte	parambuf[128];
byte	sum;
byte	cmd1;
byte	cmd2;

void serial_sync_write_bits(word data, int bits)
{
	int	i;

	for (i = 0; i < bits; i++) {
		bitop(CLOCK, 1);
		bitop(DATA, data & 1);
		WAIT_MICROSEC(syncwait * 8);
		bitop(CLOCK, 0);
		WAIT_MICROSEC(syncwait * 8);
		data >>= 1;
	}
}

void serial_sync_write()
{
	byte	count;
	byte	bytecount;
	byte	i, j;
	byte	tbits;
	byte	data;
	byte	etx;

	count = cmd2 & 0x7f;
	bytecount = (count + 7) / 8;

	for (i = 0; i < bytecount; i++) {
		sum += parambuf[i] = s_getc();
	}
#ifdef USESUMETX
	sum += s_getc();
	etx = s_getc();
	if (etx != ETX) {
		MESSAGE("ETX error");
		s_putc(NAK);
		return;
	}
	if (sum != 0) {
		MESSAGE("sum error");
		s_putc(NAK);
		return;
	}
	MESSAGE("checksum/ETX are accepted");
#endif

	WRITER.DDR = 0xff;

	j = 0;
	while (count > 0) {
		MESSAGE("writing data");
		tbits = count;
		if (tbits > 8)
			tbits = 8;
		data = parambuf[j];
		serial_sync_write_bits(data, tbits);
		count -= tbits;
		j++;
		MESSAGE("wrote");
	}
	MESSAGE("done");
	s_putc(ACK);
	WRITER.DDR = 0xfe;
}

void serial_sync_read()
{
	byte	count;
	byte	bytecount;
	byte	i, j;
	byte	tbits;
	byte	data;

	count = cmd2 & 0x7f;
	bytecount = (count + 7) / 8;
	s_putc(ACK);

	WRITER.DDR = 0xfe;
	for (j = 0; j < bytecount; j++) {
		tbits = count & 0x07;
		if (tbits == 0)
			tbits = 8;
		data = 0;
		for (i = 0; i < tbits; i++) {
			byte r;

			data >>= 1;
			bitop(CLOCK, 1);
			WAIT_MICROSEC(syncwait * 8);
			r = bitget(DATA);
			data |= r ? 0x80 : 0x00;
			bitop(CLOCK, 0);
			WAIT_MICROSEC(syncwait * 8);
		}
		for (; i < 8; i++) {
			data >>= 1;
		}
		parambuf[j] = data;
		count -= tbits;
	}
	sum = 0;
	for (j = 0; j < bytecount; j++) {
		s_putc(parambuf[j]);
		sum -= parambuf[j];
	}
#ifdef USESUMETX
	s_putc(sum);
	s_putc(ETX);
#endif
}

byte	infodata[] = {
	0, IF2_BS_128, IF3_VERSION_1, IF4_HARDWARE_H8_PROTOTYPE
};

void info()
{
	byte	i;

	s_putc(ACK);
	sum = 0;
	for (i = 0; i < sizeof(infodata); i++) {
		s_putc(infodata[i]);
		sum -= infodata[i];
	}
#ifdef USESUMETX
	s_putc(sum);
	s_putc(ETX);
#endif
}

void line_control()
{
	byte	count;
	byte	etx;
	int		i;

	count = cmd2 & 0x7f;

	for (i = 0; i < count; i++) {
		sum += parambuf[i] = s_getc();
	}
#ifdef USESUMETX
	sum += s_getc();
	etx = s_getc();
	if (etx != ETX) {
		MESSAGE("ETX error");
		s_putc(NAK);
		return;
	}
	if (sum != 0) {
		MESSAGE("sum error");
		s_putc(NAK);
		return;
	}
#endif	
	MESSAGE("Start control");
	for (i = 0; i < count; i++) {
		WRITER.DR.BYTE = parambuf[i];
		WAIT_MICROSEC(linewait * 8);
		MESSAGE("looping");
	}
	s_putc(ACK);
}

void port_control()
{
	// I/O bits from PC is 0=Out 1=In : H8 DDR bits are their complement.
	// bit6/7 are always output
	cmd2 &= 0x3f;
	WRITER.DDR = ~cmd2;
	MESSAGE("Port control OK");
	s_putc(ACK);
}

void set_sync_wait()
{
#ifndef SLOWSERIAL
	syncwait = cmd2 & 0x7f;
#endif
	s_putc(ACK);
}

void set_line_wait()
{
#ifndef SLOWSERIAL
	linewait = cmd2 & 0x7f;
#endif
	s_putc(ACK);
}

byte	algwords;

void write_alg()
{
	byte	count;
	byte	etx;
	byte	words;
	int		i;

	count = cmd2 & 0x7f;
	if (count == 0)
		count = 128;
	if (count > 128) {
		s_putc(NAK);
		return;
	}
	if (count & 0x0f) {
		s_putc(NAK);
		return;
	}
	words = count / 2;

	for (i = 0; i < count; i++) {
		sum += parambuf[i] = s_getc();
	}
#ifdef USESUMETX
	sum += s_getc();
	etx = s_getc();
	if (etx != ETX) {
		MESSAGE("ETX error");
		s_putc(NAK);
		return;
	}
	if (sum != 0) {
		MESSAGE("sum error");
		s_putc(NAK);
		return;
	}
#endif
	byte wc;
	word w;

	wc = algwords;
	for (i = 0; i < words; i++) {
		serial_sync_write_bits(PGM_LOAD_P, 6);
		WAIT_MICROSEC(1);
		w = parambuf[i * 2 + 1];
		w <<= 8;
		w |= parambuf[i * 2];
		w <<= 1;

		serial_sync_write_bits(w, 16);
		WAIT_MICROSEC(1);

		if (--wc == 0) {
			serial_sync_write_bits(PGM_BEGIN_P_XT, 6);
			WAIT_MILLISEC(3);

			serial_sync_write_bits(PGM_END_P, 6);
			WAIT_MICROSEC(1);
			wc = algwords;
		}
		serial_sync_write_bits(PGM_INCREMENT, 6);
		WAIT_MICROSEC(1);
	}
	s_putc(ACK);
}

void binio()
{

	while (1) {
		sum = 0;
		cmd1 = s_getc();
		if (cmd1 == SYN) {
			s_putc(SYN);
			continue;
		}
		sum += cmd1;

		if (!(cmd1 & 0x80)) {
			MESSAGE("BAD CMD1");
			s_putc(NAK);
			continue;
		}
		cmd2 = s_getc();
		if (cmd2 == SYN) {
			s_putc(SYN);
			continue;
		}
		if (!(cmd2 & 0x80)) {
			MESSAGE("MSB of cmd2 is not 1");
			s_putc(NAK);
			continue;
		}
		sum += cmd2;

		switch (cmd1) {
		case WP_INFO:	// mandatory
			info();
			break;
		case WP_SETSYNCWAIT:
			set_sync_wait();
			break;
		case WP_SETLINEWAIT:
			set_line_wait();
			break;
		case WP_SERIALSYNCWRITE:
			serial_sync_write();
			break;
		case WP_SERIALSYNCREAD:
			serial_sync_read();
			break;
		case WP_PORTCONTROL:
			port_control();
			break;
		case WP_LINECONTROL:
			line_control();
			break;
		default:
			MESSAGE("INVALID COMMAND");
			s_putc(NAK);
		}
	}
}

int main()
{
	WRITER.DR.BYTE = 0x00;
	WRITER.DDR = 0xfe;

#if 1
	WAIT_MICROSEC(linewait * 8);
	binio();
	return 0;
#endif

#ifdef LINETEST
	linetest();
	return 0;
#endif

#if 0
	waitkey("program wait>");

	bitop(PGM, 1);
	bitop(MCLR, 1);
	wait(THLD0);

	write_command6(COMMAND_LOAD_CONFIGURATION);	
	wait(TDLY2);
	write_data14(0);	// DUMMY
	wait(TDLY1);

	for (i = 0; i < 10; i++) {
		write_command6(COMMAND_READ_DATA_FROM_PROGRAM_MEMORY);
		wait(TDLY2);
		r = read_data14();
		wait(TDLY1);
		s_puts("Result: ");
		printhex16(r);
		s_puts("\n");
		write_command6(COMMAND_INCREMENT_ADDRESS);
		wait(10000);
	}
	WRITER.DDR = 0xfe;
	WRITER.DR.BYTE = 0x00;
	return 0;
#endif
}

