// BK1088 Control Routine For WinAVR
// (c) 2013 Eleken 
// rev_c1815@yahoo.co.jp

#include "main.h"
#include "bk1088.h"

// local functions (I2C)
static void I2C_SendStartBit(void){
	CBI(BK1088_DDR, BK1088_DATA); // DATA -> H
	_delay_us(2);
	SBI(BK1088_PORT, BK1088_SCLK);// SCLK -> H
	_delay_us(2);
	SBI(BK1088_DDR, BK1088_DATA); // DATA -> L
	_delay_us(20);
	CBI(BK1088_PORT, BK1088_SCLK);// SCLK -> L
	_delay_us(20);
	CBI(BK1088_DDR, BK1088_DATA); // DATA -> H
	_delay_us(20);
}
static void I2C_SendStopBit(void){
	CBI(BK1088_PORT, BK1088_SCLK);// SCLK -> L
	_delay_us(2);
	SBI(BK1088_DDR, BK1088_DATA); // DATA -> L
	_delay_us(2);
	SBI(BK1088_PORT, BK1088_SCLK);// SCLK -> H
	_delay_us(2);
	CBI(BK1088_DDR, BK1088_DATA); // DATA -> H
	_delay_us(20);
//	CBI(BK1088_PORT, BK1088_SCLK);
//	_delay_us(2);
}

// Send 1bit ( d=[0,1]  )
static void I2C_SendBit(uint8_t d){
	if(d & 0x01){
		CBI(BK1088_DDR, BK1088_DATA); 
	}else{
		SBI(BK1088_DDR, BK1088_DATA); 
	}
	_delay_us(2);
	SBI(BK1088_PORT, BK1088_SCLK);// SCLK -> H
	_delay_us(2);
	CBI(BK1088_PORT, BK1088_SCLK);// SCLK -> L
	_delay_us(2);
}

// Send Ack/Nack ( d=[0:ack,1:nack]  )
static void I2C_SendAck(uint8_t d){
	CBI(BK1088_PORT, BK1088_SCLK);// SCLK -> L
	if(d & 0x01){
		CBI(BK1088_DDR, BK1088_DATA); 
	}else{
		SBI(BK1088_DDR, BK1088_DATA); 
	}
	_delay_us(2);
	SBI(BK1088_PORT, BK1088_SCLK);// SCLK -> H
	_delay_us(2);
	CBI(BK1088_PORT, BK1088_SCLK);// SCLK -> L
	_delay_us(2);
}
static uint8_t I2C_RcvAck(void){
	uint8_t ret = 0;
	CBI(BK1088_DDR, BK1088_DATA); // DATA -> H
	_delay_us(2);
	SBI(BK1088_PORT, BK1088_SCLK);// SCLK -> H
	_delay_us(2);
	if(BK1088_PIN & _BV(BK1088_DATA)){ // '1'
		ret = 1;
	}
	CBI(BK1088_PORT, BK1088_SCLK);// SCLK -> L
	_delay_us(2);
	return ret;
}


// Send 1byte 
static void I2C_SendByte(uint8_t d){
	uint8_t i;
	for(i=0; i<8; i++){
		if(d & 0x80){
			CBI(BK1088_DDR, BK1088_DATA); 
		}else{
			SBI(BK1088_DDR, BK1088_DATA); 
		}
		_delay_us(2);
		SBI(BK1088_PORT, BK1088_SCLK); // SCLK -> H
		d <<= 1;
		_delay_us(2);
		CBI(BK1088_PORT, BK1088_SCLK); // SCLK -> L
		_delay_us(2);
	}
}

// Receive 1byte
static void I2C_RcvByte(uint8_t* d){
	uint8_t i;
	*d = 0;
	CBI(BK1088_DDR, BK1088_DATA);  // DATA -> H
	for(i=0; i<8; i++){
		(*d) <<= 1;
		SBI(BK1088_PORT, BK1088_SCLK); // SCLK -> H
		_delay_us(20);
		if(BK1088_PIN & _BV(BK1088_DATA)){ // '1'
			(*d) |= 1;
		}
		CBI(BK1088_PORT, BK1088_SCLK); // SCLK -> L
		_delay_us(20);
	}
}

// Send Data
void BK1088_Send(uint8_t addr, uint16_t data){
	// Note: ACK bit is sended from slave and should be checked by host, but here we do not check
	I2C_SendStartBit();
	
	I2C_SendByte( 0x80 ); // Device ID
	I2C_RcvAck(); // Ack
	
	I2C_SendByte( addr<<1 ); // Address
	I2C_RcvAck(); // Ack
	
	I2C_SendByte( (uint8_t)(data>>8) ); // Data(H)
	I2C_RcvAck(); // Ack
	
	I2C_SendByte( (uint8_t)(data&0xFF) ); // Data(L)
	I2C_RcvAck(); // NAck
	
	I2C_SendStopBit();
	_delay_ms(1);
}

// Receive Data
void BK1088_Receive(uint8_t addr, uint16_t* data){
	// Note: in this procedure, distinct from standard I2C protocol, ACK bit must be sended from host controller
	I2C_SendStartBit();
	
	I2C_SendByte( 0x80 ); // Device ID
	I2C_RcvAck(); // Ack
	
	I2C_SendByte( (addr<<1) | 1); // Address
	I2C_RcvAck(); // Ack
	
	uint8_t temp;
	
	I2C_RcvByte( &temp ); // Data(H)
	I2C_SendAck(0); // Ack

	*data = ((uint16_t)temp)<<8;
	
	I2C_RcvByte( &temp ); // Data(L)
	I2C_SendAck(1); // NAck
	
	*data |= (uint16_t)temp;
	
	I2C_SendStopBit();
	_delay_us(3);
}

#include "bk1088_reg.h"

static void BK1088_SetBit(uint8_t addr, uint16_t bit, uint8_t value){
	uint16_t temp;
	BK1088_Receive(addr, &temp);
	temp = (temp & (0xFFFF-bit)) | ((value)? bit : 0);
	BK1088_Send(addr, temp);
}
// Initialize
uint8_t BK1088_Init(void){
	uint8_t i;
	uint16_t temp;
	// init port 
	BK1088_DDR  = (BK1088_DDR & BK1088_MASK)  | (1<<BK1088_SCLK);
	BK1088_PORT = (BK1088_PORT & BK1088_MASK) | (1<<BK1088_SCLK);
	
	// check chip ID
	BK1088_Receive(1, &temp);
	if(temp != 0x1080) return 1;
	
//	for(i=0; i<250; i++) _delay_ms(2);
	
	// send init data
	for(i=2; i<41; i++){
		//if(i==25) continue;
		BK1088_Send(i, pgm_read_word(&(initData_FM[i])));
	}
	
	for(i=0; i<250; i++) _delay_ms(2);
	
	BK1088_SetMode(CHANNEL_FM);
	

	
	return 0;
}

// Send new receive channel #
// freq = BAND + channel * SPACE
// e.g. (mode=0x0E, channel=48) -> freq = 522 + 48*9 = 954[khz]
void BK1088_SetChannel(uint16_t channel){
	uint8_t i;
	// Set Seek To 0
	BK1088_SetBit(0x02, 0x0100, 0);

	// Set Channel
	BK1088_Send(0x03, channel);
	_delay_ms(1);
	BK1088_Send(0x03, channel | 0x8000);
	for(i=0; i<50; i++) _delay_ms(1);
}
// Get current receive channel #
uint16_t BK1088_GetCurrentChannel(void){
	uint16_t temp;
	
	BK1088_Receive(0x0B, &temp); // Get Read Channel
	return temp & 0x7FFF;
}
uint8_t BK1088_IsTuned(void){
	uint16_t temp;
	
	BK1088_Receive(0x03, &temp);
	if(temp & 0x8000) return 1;
	else return 0;
}

// Seek next(prev) channel (non-blocking)
// direction: 1:FWD, 0:REV
void BK1088_SeekNext(uint8_t direction){
	BK1088_SetBit(0x03, 0x8000, 0);
	BK1088_SetBit(0x02, 0x0400, 0);
	if(direction & 0x01){
		BK1088_SetBit(0x02, 0x0200, 1);
	}else{
		BK1088_SetBit(0x02, 0x0200, 0);
	}
	// Seek Start
	BK1088_SetBit(0x02, 0x0100, 0);
	BK1088_SetBit(0x02, 0x0100, 1);
	
}
// Seek next(prev) channel (blocking)
// direction: 1:FWD, 0:REV
void BK1088_SeekNext2(uint8_t direction){
	BK1088_SetBit(0x03, 0x8000, 0);
	BK1088_SetBit(0x02, 0x0400, 1);
	if(direction & 0x01){
		BK1088_SetBit(0x02, 0x0200, 1);
	}else{
		BK1088_SetBit(0x02, 0x0200, 0);
	}
	// Seek Start
	BK1088_SetBit(0x02, 0x0100, 0);
	BK1088_SetBit(0x02, 0x0100, 1);
	
	// polling
	for(;;){
		uint16_t d;
		BK1088_Receive(0x0A, &d);
		if(d & 0x4000) break; // tuned
		if(g_swState){
			g_swState = 0;
			// Seek Stop
			BK1088_SetBit(0x02, 0x0100, 0);
			break;
		}
	}
}

// mode : 0bxxxx[xx][xx]
//          |   BAND SPACE
//          +Mode: 1:FM, 0:AM
//               mode   BAND   SPACE
// FM(JAPAN) ->  0x8A   76MHz  0.1MHz
// MW(JAPAN) ->  0x0E   522kHz  9kHz
// LW        ->  0x00   153kHz  1kHz
// SW        ->  0x09  2300kHz  5kHz
void BK1088_SetMode(uint8_t mode){
	uint16_t temp;
	
	// Register fix
	// Reg #16-#26, #35-40
	if(mode & 0x80){ // FM
		uint8_t i;
		BK1088_Send(8, pgm_read_word(&(initData_FM[8])));
		for(i=16; i<=24; i++){
			BK1088_Send(i, pgm_read_word(&(initData_FM[i])));
		}
		for(i=26; i<=40; i++){
			BK1088_Send(i, pgm_read_word(&(initData_FM[i])));
		}
		for(i=0; i<100; i++) _delay_ms(2);
		// send seek threshold	
		BK1088_Receive(0x05, &temp);
		temp = (temp & 0b0000000111111111) | ((uint16_t)(g_radioSettings.RSSI_thre_FM) << 9);
		BK1088_Send(0x05, temp);
		BK1088_Receive(0x06, &temp);
		temp = (temp & 0b1111100000001111) | ((uint16_t)(g_radioSettings.SNR_thre_FM) << 4);
		BK1088_Send(0x06, temp);
	}else{ // AM
		uint8_t i;
		BK1088_Send(8, pgm_read_word(&(initData_AM[8])));
		for(i=16; i<=24; i++){
			BK1088_Send(i, pgm_read_word(&(initData_AM[i])));
		}
		
		if(mode == 0x09){//sw
			BK1088_Send(24, 0xB41C);
		}else{
			BK1088_Send(24, 0x341C);
		}
		
		for(i=26; i<=40; i++){
			BK1088_Send(i, pgm_read_word(&(initData_AM[i])));
		}
		for(i=0; i<100; i++) _delay_ms(2);
		// send seek threshold
		BK1088_Receive(0x05, &temp);
		temp = (temp & 0b0000000111111111) | ((uint16_t)(g_radioSettings.RSSI_thre_AM) << 9);
		BK1088_Send(0x05, temp);
		BK1088_Receive(0x06, &temp);
		temp = (temp & 0b1111100000001111) | ((uint16_t)(g_radioSettings.SNR_thre_AM) << 4);
		BK1088_Send(0x06, temp);
	}
	
	// Set mode to FM
	BK1088_SetBit(0x07, 0x2000, !(mode & 0x80));
	// set [band, space]
	BK1088_Receive(0x05, &temp);
	temp = (temp & 0b1111111000011111) | ((uint16_t)(mode&0x0F) << 5);
	BK1088_Send(0x05, temp);
	
}

// Receive status from Tuner IC into g_radioStatus
void BK1088_UpdateStatus(void){
	uint16_t temp;
//	g_radioStatus.curChannel = BK1088_GetCurrentChannel();

	BK1088_Receive(26, &temp);
	g_radioStatus.varactor = temp;
	
	BK1088_Receive(0x09, &temp);
	g_radioStatus.SNR  = temp & 0x7F;
	BK1088_Receive(0x0A, &temp);
	g_radioStatus.RSSI = temp & 0x7F;

//	BK1088_Receive(0x0A, &temp);
	g_radioStatus.status = 0;
	uint8_t thre = g_radioSettings.SNR_thre_AM;
	if(g_radioSettings.mode & 0x80) thre = g_radioSettings.SNR_thre_FM;
	if(g_radioStatus.SNR >= thre) {
		g_radioStatus.status |= 1; // TUNE
		if(temp & 0x0080) { // STEREO
			g_radioStatus.status |= 2;
		}
		
		if(g_radioSettings.mode & 0x80){ // FM
			BK1088_Receive(0x02, &temp);
		//	if(g_radioStatus.status & 0x02){ // stereo
		//		BK1088_Send(0x02, temp | 0x0800); // set force stereo
		//	}else{
		//		BK1088_Send(0x02, temp & 0xF7FF); // clear force stereo
		//	}
		}
	}
	
}
void BK1088_SetVolume(void){
	uint16_t temp;
	BK1088_Receive(0x05, &temp);
	temp = (temp & 0xFFE0) | (g_radioSettings.vol & 0x1F);
	BK1088_Send(0x05, temp);
}

// FM(JAPAN) ->  0x8A   76MHz  0.1MHz
// MW(JAPAN) ->  0x0E   522kHz  9kHz
// LW        ->  0x00   153kHz  1kHz
// SW        ->  0x09  2300kHz  5kHz

// Output: (AM Mode->) Frequency in kHz(0-65,535)
//       : (FM Mode->) Frequency in 100kHz(760-1080)
uint16_t BK1088_GetCurrentFreq(void){

	uint16_t ch = BK1088_GetCurrentChannel();
	uint16_t ret = 0;
	switch(g_radioSettings.mode){
	case CHANNEL_MW: // MW
		ret = 522 + ch*9;
		break;
	case CHANNEL_SW: // SW
		ret = 2300 + ch*5;
		break;
	case CHANNEL_LW: // LW
		ret = 153 + ch*1;
		break;
	default: // FM
		ret = 760 + ch*1;
		break;
	}
	return ret;
}

// Input : (AM Mode->) Frequency in kHz(0-65,535)
//       : (FM Mode->) Frequency in 100kHz(760-1080)
uint16_t FreqToChannel(uint16_t freq){
	return 0;
}

