// Digital Radio / for ATMEGA328P
// 2013.2.14 Eleken

// main clock : 8.0 MHz

// [VCC, GND, MOSI, MISO, SCK, RST]

#include "main.h"
#include "lcd3300.h"
#include "bk1088.h"

char tbuf[22];

struct tBk1088Settings g_radioSettings;
struct tBk1088Status   g_radioStatus;
char g_channelDescription[64];

// Preset list
// format: {use_as_preset[1(non-preset), 2(preset)], channel+1 (little-endian), channel description (null-terminated; <22 letters)}
// must place in ascending order
uint8_t g_presetFM [] PROGMEM = {
0,0, // start
2, LB((761-760)+1), HB((761-760)+1), 'I','n','t','e','r',' ','F','M', '\0', //
2, LB((765-760)+1), HB((765-760)+1), 'U','-','A','i','r','(','C','A','T','V',')', '\0', // 
1, LB((771-760)+1), HB((771-760)+1), 'U','-','A','i','r', '\0', // 
2, LB((775-760)+1), HB((775-760)+1), 'N','A','C','K','5','(','C','A','T','V',')', '\0', // 
1, LB((780-760)+1), HB((780-760)+1), 'b','a','y',' ','F','M', '\0', // 
2, LB((783-760)+1), HB((783-760)+1), 'F','M',' ','Y','O','K','O','H','A','M','A','(','C','A','T','V',')', '\0', // 
1, LB((789-760)+1), HB((789-760)+1), 'K','a','t','s','u','s','h','i','k','a',' ','F','M', '\0', // 
2, LB((791-760)+1), HB((791-760)+1), 'J','-','W','A','V','E','(','C','A','T','V',')', '\0', // 
2, LB((795-760)+1), HB((795-760)+1), 'N','A','C','K','5', '\0', // 
2, LB((800-760)+1), HB((800-760)+1), 'T','O','K','Y','O',' ','F','M', '\0', // 
2, LB((804-760)+1), HB((804-760)+1), 'N','H','K',' ','T','O','K','Y','O','(','C','A','T','V',')', '\0', // 
2, LB((813-760)+1), HB((813-760)+1), 'J','-','W','A','V','E', '\0', // 
2, LB((825-760)+1), HB((825-760)+1), 'N','H','K',' ','T','O','K','Y','O', '\0', // 
2, LB((832-760)+1), HB((832-760)+1), 'T','O','K','Y','O',' ','F','M','(','C','A','T','V',')', '\0', // 
2, LB((856-760)+1), HB((856-760)+1), 'b','a','y',' ','F','M','(','C','A','T','V',')', '\0', // 
0,0,0,// end
};
uint8_t g_presetAM [] PROGMEM = {
0,0, // start
2, LB((594-522)/9+1), HB((594-522)/9+1), 'J','O','A','K',' ','/',' ','N','H','K','1', '\0', // 
2, LB((693-522)/9+1), HB((693-522)/9+1), 'J','O','A','B',' ','/',' ','N','H','K','2', '\0', // 
2, LB((810-522)/9+1), HB((810-522)/9+1), '-','-','-','-',' ','/',' ','A','F','N',' ','T','O','K','Y','O', '\0', //
2, LB((954-522)/9+1), HB((954-522)/9+1), 'J','O','K','R',' ','/',' ','T','B','S', '\0', //
2, LB((1134-522)/9+1), HB((1134-522)/9+1), 'J','O','Q','R',' ','/',' ','N','C','B', '\0', 
1, LB((1197-522)/9+1), HB((1197-522)/9+1), 'J','O','Y','F',' ','/',' ','I','B','S', '\0', //
2, LB((1242-522)/9+1), HB((1242-522)/9+1), 'J','O','L','F',' ','/',' ','L','F', '\0', //
2, LB((1422-522)/9+1), HB((1422-522)/9+1), 'J','O','R','F',' ','/',' ','R','F', '\0', //
1, LB((1530-522)/9+1), HB((1530-522)/9+1), 'J','O','X','F',' ','/',' ','C','R','T', '\0', //
0,0,0,// end
};
uint8_t g_presetSW [] PROGMEM = {
0,0, // start
1,  LB((3250-2300)/5+1), HB((3250-2300)/5+1), 'R','a','d','i','o',' ','P','y','o','n','g','y','a','n','g',' ','1', '\0', // 
2,  LB((3925-2300)/5+1), HB((3925-2300)/5+1), 'N','i','k','k','e','i',' ','1','(','J','O','Z','1',')', '\0', // 
1,  LB((5955-2300)/5+1), HB((5955-2300)/5+1), 'C','h','i','n','a',' ','R','a','d','i','o',' ','I','t','n','l','.', '\0', // 
2,  LB((6055-2300)/5+1), HB((6055-2300)/5+1), 'N','i','k','k','e','i',' ','1','(','J','O','Z','2',')', '\0', // 
2,  LB((6070-2300)/5+1), HB((6070-2300)/5+1), 'V','o','i','c','e',' ','o','f',' ','K','o','r','e','a',' ','1', '\0', // 
1,  LB((6250-2300)/5+1), HB((6250-2300)/5+1), 'R','a','d','i','o',' ','P','y','o','n','g','y','a','n','g',' ','2', '\0', // 
1,  LB((7205-2300)/5+1), HB((7205-2300)/5+1), 'C','h','i','n','a',' ','R','a','d','i','o',' ','I','t','n','l','.', '\0', // 
2,  LB((7235-2300)/5+1), HB((7235-2300)/5+1), 'V','o','i','c','e',' ','o','f',' ','R','u','s','s','i','a', '\0', // 
2,  LB((7325-2300)/5+1), HB((7325-2300)/5+1), 'C','h','i','n','a',' ','R','a','d','i','o',' ','I','t','n','l','.', '\0', // 
2,  LB((9595-2300)/5+1), HB((9595-2300)/5+1), 'N','i','k','k','e','i',' ','1','(','J','O','Z','3',')', '\0', // 
2,  LB((9650-2300)/5+1), HB((9650-2300)/5+1), 'V','o','i','c','e',' ','o','f',' ','K','o','r','e','a',' ','2', '\0', // 
2,  LB((9735-2300)/5+1), HB((9735-2300)/5+1), 'R','a','d','i','o',' ','T','a','i','w','a','n',' ','I','t','n','l','.', '\0', // 
2,  LB((11620-2300)/5+1), HB((11620-2300)/5+1), 'C','h','i','n','a',' ','R','a','d','i','o',' ','I','t','n','l','.', '\0', // 
1,  LB((11680-2300)/5+1), HB((11680-2300)/5+1), 'K','C','B','S',' ','P','y','o','n','g','y','a','n','g', '\0', // 
2,  LB((11865-2300)/5+1), HB((11865-2300)/5+1), 'V','o','i','c','e',' ','o','f',' ','K','o','r','e','a',' ','3', '\0', // 
1,  LB((13620-2300)/5+1), HB((13620-2300)/5+1), 'C','h','i','n','a',' ','R','a','d','i','o',' ','I','t','n','l','.', '\0', // 
0,0,0,// end
};


/* Function */
void FormatMsg(char* output, uint8_t length, const char* format, unsigned int x1);

/* EEPROM */
uint8_t eep_dummy __attribute__((section(".eeprom")));
uint8_t eep_area[sizeof(g_radioSettings)] __attribute__((section(".eeprom")));

static void LoadEEP(void){ // EEPROM -> RAM
	uint8_t i;
	uint8_t* p = (uint8_t*)(&g_radioSettings);
	for(i=0; i<sizeof(g_radioSettings); i++){
		uint8_t a = eeprom_read_byte(&(eep_area[i]));
		if(a != 0xFF) p[i] = a;
	}
}
static void SaveEEP(void){ // RAM -> EEPROM 
	uint8_t i;
	uint8_t* p = (uint8_t*)(&g_radioSettings);
	for(i=0; i<sizeof(g_radioSettings); i++){
		eeprom_write_byte(&(eep_area[i]), p[i]);
	}
}

/* Timer */
volatile uint8_t g_swState = 0;
volatile uint8_t g_PWMValue = 1;
volatile uint16_t g_timeAfterSwitch = 0; // in second, 0-65535
uint8_t g_backlight; // Backlight ON(1)/OFF(0)/Weak(2)
uint8_t g_blWeakBySec; // Backlight power down threshold [sec]
uint8_t g_blOffBySec; // Backlight power off  threshold [sec]
/* 1: "SEL", 2: "-", 3: "+", 4: "POW", 8: (long-pushed) 16: (being long-pushed) */
// INT0
ISR(INT0_vect){
}

// timer2 : 1s
ISR(TIMER2_OVF_vect){
	// start adc
//	while(ADCSRA & 0b01000000);
//	ADMUX   = (ADMUX & 0xF0) | 0b00000110; // set channel to ADC6
//	ADCSRA |= 0b01000000; // start conversion

}
// ADC conversion complete
ISR(ADC_vect){
/*
	if((ADMUX & 0x0F) == 0b00000110){ // channel was ADC6
		uint16_t ADC_criterion;
		if(g_backlight == 1) ADC_criterion = 558; // 1024 * (object[V]) / 1.1   (0.6V)
		else ADC_criterion = 276; // 1024 * (object[V]) / 1.1   (0.6V)
		
		if(ADC < ADC_criterion){
			if(g_PWMValue < 240) g_PWMValue += 1;
		}else{
			if(g_PWMValue > 1) g_PWMValue -= 1;
		}
		
		if(g_backlight){
			OCR0B = g_PWMValue;
			TCCR0A = 0b00100011; // 8bit HS-PWM, OC0B -> Non-inversed output
		}else{
			OCR0B = g_PWMValue = 0;
			TCCR0A = 0b00000011; // 8bit HS-PWM, OC0B -> cutoff
		}
	}
	*/
}

// timer1 : 100ms
ISR(TIMER1_COMPA_vect){
	static uint8_t sw_old;
	static uint8_t longpush_cnt;
	static uint8_t s;
	static uint8_t cnt;
	
	{ // check switch state
		uint8_t sw = (SW_PORT & SW_PORT_MASK);
		
		if((sw != SW_PORT_MASK) && (longpush_cnt > LONGPUSH_WEIGHT)){ // being long-pushed
			g_swState = SW_PUSHING | s;
			g_timeAfterSwitch = 0;
		}else if(sw_old != sw){ // switch state has changed
			if(sw == SW3_PUSHED){//B
				s = 3;
			}else if(sw == SW4_PUSHED){//C
				s = 4;
			}else if(sw == SW2_PUSHED){//SEL
				s = 2;
			}else if(sw == SW1_PUSHED){//POW
				s = 1;
			}else if(sw == SW_PORT_MASK){//off
				if(longpush_cnt == 0) {
					//sw_state = 0; // too short
				}else if(longpush_cnt <= LONGPUSH_WEIGHT) {// < 500ms
					g_swState = s; // short push
					g_timeAfterSwitch = 0;
				}else {
					g_swState = SW_LONG | s; // long push
					g_timeAfterSwitch = 0;
				}
				s = 0;
			}else {
				s = 0;
			}
		}
		
		if(sw != SW_PORT_MASK){ // Pushed
			longpush_cnt ++;
		}else{
			longpush_cnt = 0;
		}
		sw_old = sw;
	}
	
	cnt++;
	if(cnt > 10) {
		cnt = 0;
		g_timeAfterSwitch ++;
	}
}

void InitPort(void){
	
	// init clock
	CLKPR = 0x80;
	CLKPR = 0x00; // CLKPS = 0 (8MHz)
	
	MCUSR = 0;
	
	uint8_t i;
	for(i=0; i<100; i++) _delay_ms(2);
	
	// init pins
	PORTB = 0b00111111;
	DDRB  = 0b00000000;
	
	PORTC = 0b00100011;
	DDRC  = 0b00011101;
	
	PORTD = 0b10110111;
	DDRD  = 0b01111100;
	
}

// Initialize
uint8_t Init(void){
	
	cli();
	InitPort();
	
	// init SPI (clock:5.0MHz)
//	SPCR = 0b01010000;// SPI mode 0
//	SPSR = 0b00000001;
	
	// init timer
	// timer0 (PWM)
	/*
	TCCR0A = 0b00100011; // 8bit HS-PWM, OC0B -> Non-inversed output
	TCCR0B = 0b00000001; // Clock -> clk/1
	OCR0B  = g_PWMValue;
	TIMSK0 = 0b00000000;
	TCNT0  = 0;
	*/
	
	// timer1 (100 ms)
	TCCR1A = 0b00000000; // CTC
	TCCR1B = 0b00001101; // clk/1024
	OCR1A = 781; // when clock is 8 MHz
	TIMSK1 = 0b00000010; // Compare match interrupt enabel
	TCNT1 = 0;
	
	// timer2 (1 ms)
	TCCR2A = 0b00000000;
	TCCR2B = 0b00000011; // clk/32
	TIMSK2 = 0b00000001;
	ASSR   = 0;
	TCNT2  = 0;
	
	// ADC
	/*
	ADMUX  = 0b11000110; // (Aref = 1.1V, ADC6)
	ADCSRA = 0b10001101; // clk/32, auto trigger disable, interrupt enable
	ADCSRB = 0b00000000;
	ADCSRA |= 0b01000000; // start conversion
	*/
	
	// disable TWI, USART0, Acomp
//	PRR  = 0b11000010;
//	ACSR = 0b10000000;
	

	// init variables
//	eep_var_load();

	// init LCD controller
	LCD3300_Init();
	// init Radio module
	uint8_t ret = BK1088_Init();
	
	g_backlight = 1;
	g_blWeakBySec = 3;
	g_blOffBySec  = 6;

	sei();
	
	if(ret) return 1;
	// Show Logo
	
	return 0;
	
}

// Example: FormatMsg(a, 10, "x = [%d]", 30) => a: "x = [30]  "
// buffer size must be at least (length + 1)
// This is debug version: not check output buffer length
void FormatMsg(char* output, uint8_t length, const char* format, unsigned int x1){
	uint8_t i, j=0;
	char temp[8];
	for(i=0; i<128; i++){
		if(j >= length) break;
		output[j] = format[i];
		if(format[i] == '\0') break;
		else if(format[i] == '%'){
			uint8_t base = 0;
			if(format[i+1] == 'd'){
				base = 10;
				utoa(x1, temp, base);
			}
			else if(format[i+1] == 'x'){
				base = 16;
				utoa(x1, temp, base);
			}
			if(base){
				uint8_t k=0;
				while(temp[k] != '\0'){
					output[j++] = temp[k++];
					if(j >= length) break;
				}
				i++;
			}
		}else{
			j++;
		}
	}
	while(j<length) output[j++] = ' ';
	output[j] = '\0';
}

// Preset Managing Function (Preset Manager)

// Get preset information whose freq is closest and greather than curChannel 
// Return value: 0(successful)
// direction: 0(REV), 1(FWD)
uint8_t PMGetNextPreset(uint8_t band, uint16_t curChannel, uint16_t* newChannel, char* text, uint8_t direction){
	uint16_t idx = 2;
	uint16_t idx_old = 0;
	prog_uchar * plist = g_presetFM;
	if(band == CHANNEL_MW){ // AM
		plist = g_presetAM;
	}else if(band == CHANNEL_FM){
		plist = g_presetFM;
	}else if(band == CHANNEL_SW){
		plist = g_presetSW;
	}else{
		return 1;
	}
	for(;;){
		uint8_t useAsPreset = pgm_read_byte(&(plist[idx+0])) - 1;
		// check freq & cond
		if(useAsPreset){
			(*newChannel) = pgm_read_word(&(plist[idx+1]));
			if((*newChannel) > curChannel) {
				if(direction == 0){ // REV
					if(idx_old == 0) return -1;
					(*newChannel) = pgm_read_word(&(plist[idx_old+1])) - 1;
					strcpy_P(text, (PGM_P)&(plist[idx_old+3]));
					return 0;
				}else{ // FWD
					(*newChannel) --;
					strcpy_P(text, (PGM_P)&(plist[idx+3]));
					return 0;
				}
			}else if((*newChannel) == 0){
				break;
			}
			 idx_old = idx;
		}
		// seek next list pointer
		idx += 3;
		while(pgm_read_byte(&plist[idx++]) != 0);
	}
	if((direction == 0) && (idx_old != 0)) {
		(*newChannel) = pgm_read_word(&(plist[idx_old+1])) - 1;
		strcpy_P(text, (PGM_P)&(plist[idx_old+3]));
		return 0;
	}
	return 1;
}
uint8_t PMGetPresetDescription(uint8_t band, uint16_t curChannel, char* text){
	uint16_t idx = 2;
	uint16_t newChannel;
	prog_uchar * plist = g_presetFM;
	if(band == CHANNEL_MW){ // AM
		plist = g_presetAM;
	}else if(band == CHANNEL_FM){
		plist = g_presetFM;
	}else if(band == CHANNEL_SW){
		plist = g_presetSW;
	}else{
		text[0] = '\0';
		return 1;
	}
	for(;;){
		// check
		newChannel = pgm_read_word(&(plist[idx+1]));
		if((newChannel-1) == curChannel) {
			strcpy_P(text, (PGM_P)&(plist[idx+3]));
			return 0;
		}else if(newChannel == 0){
			break;
		}
		// seek next list pointer
		idx += 3;
		while(pgm_read_byte(&plist[idx++]) != 0);
	}
	text[0] = '\0';
	return 1;
}

static void ChangeANTSwitch(uint8_t flag){
	uint8_t i;
	if(flag){
		if(flag & 1){
			SBI(PORTC, PC4);
			SBI(PORTC, PC2);
			for(i=0; i<10; i++){
				_delay_ms(1);
			}
			CBI(PORTC, PC2);
		}else{
			CBI(PORTC, PC4);
			SBI(PORTC, PC3);
			for(i=0; i<10; i++){
				_delay_ms(1);
			}
			CBI(PORTC, PC3);
		}
		return;
	}
	if(g_radioSettings.mode == CHANNEL_SW){
		SBI(PORTC, PC4);
		SBI(PORTC, PC2);
		for(i=0; i<10; i++){
			_delay_ms(1);
		}
		CBI(PORTC, PC2);
	}else{
		CBI(PORTC, PC4);
		SBI(PORTC, PC3);
		for(i=0; i<10; i++){
			_delay_ms(1);
		}
		CBI(PORTC, PC3);
	}
}

// Show Overwrap Display
static void ShowOverwrapDisplay(void){
	static uint8_t status_old;
	// info
	if(status_old != g_radioStatus.status || 1){
		uint8_t  color;
		// show info
		
		char temp[32];
		strcpy_P(temp, PSTR("RSSI: %d  "));
		FormatMsg(tbuf, 21, temp, g_radioStatus.RSSI);
		LCD3300_DrawText(tbuf, 0,12*7, 0, 0b10110110);
		strcpy_P(temp, PSTR("SNR: %d[dB]"));
		FormatMsg(tbuf, 21, temp, g_radioStatus.SNR);
		LCD3300_DrawText(tbuf, 9*6 ,12*7, 0, 0b10110110);
		
		LCD3300_DrawBarGraph(g_radioStatus.RSSI, 0, 12*6, 0b00100101, 0b00011100);
		
		strcpy_P(tbuf, PSTR(" TUNE "));
		if(g_radioStatus.status & 0x01) color = 0b11101000;
		else color = 0b00000000;
		LCD3300_DrawText(tbuf, 0*6 ,12*1, 0b10010001, color);
		strcpy_P(tbuf, PSTR("STEREO                "));
		if(g_radioStatus.status & 0x02) color = 0b11111100;
		else color = 0b00000000;
		LCD3300_DrawText(tbuf, 6*6 ,12*1, 0b10010001, color);
		status_old = g_radioStatus.status;
		
	}
	// debug
	if(0){
		char temp[32];
		strcpy_P(temp, PSTR("Vara: %d"));
		FormatMsg(tbuf, 21, temp, g_radioStatus.varactor);
		LCD3300_DrawText(tbuf, 0, 12*4, 0b10010001, 0b11100000);
	}
	//
}
// Show Radio Display
static void ShowRadioDisplay(void){
	uint8_t tuning ;
	char temp[22];
	
	// Show Vol
	strcpy_P(temp, PSTR("Vol: %d"));
	FormatMsg(tbuf, 21, temp, g_radioSettings.vol);
	LCD3300_DrawText(tbuf, 0,12*0, COLOR_BG, 0xFF);
	
//	LCD3300_ClearDisplay();
	/*
		* maximum length of 1-line text: 128/6 = 21
		* maximum number of line : 128/12 = 10
		
		Display image:
		0	Vol:[vol] [Vbat]
		1	TUNE STEREO
		2	   : :   : 
		3	   V V   V  <------36pt. font
		4	FM ^ ^ . ^ MHz
		5	[FM TOKYO]  <------ <18 letters
		6	RSSI:___ / SNR:___
		7	[||||||||||||||||||||]
		8	
		9	 Preset Auto Manual
		    0123456789012345678901
	*/
	
	// Show Freq (16x24 font)
	if(g_radioSettings.mode == CHANNEL_FM){
		strcpy_P(tbuf, PSTR("AM"));
		LCD3300_DrawText(tbuf, 8*0, 12*3, COLOR_BG, 0b10010001);
		strcpy_P(tbuf, PSTR("FM"));
		LCD3300_DrawText(tbuf, 8*0, 12*2, COLOR_BG, COLOR_TEXT_FM);
		strcpy_P(tbuf, PSTR("MHz"));
		LCD3300_DrawText(tbuf, 8*4+64, 12*3, COLOR_BG, COLOR_TEXT_FM);
		uint16_t freq = BK1088_GetCurrentFreq();
		// freq -> text
		{
			tbuf[0] = tbuf[1] = tbuf[3] = '0';
			tbuf[2] = ':';
			tbuf[4] = '\0';
			while(freq >= 100){
				tbuf[0] ++;
				freq -= 100;
			}
			while(freq >= 10){
				tbuf[1] ++;
				freq -= 10;
			}
			tbuf[3] += freq;
		}
		// show freq
		LCD3300_DrawBigText(tbuf, 6*3, 12*2, COLOR_BG, COLOR_TEXT_FM);
	}else if(g_radioSettings.mode == CHANNEL_MW){
		strcpy_P(tbuf, PSTR("AM"));
		LCD3300_DrawText(tbuf, 8*0, 12*3, COLOR_BG, COLOR_TEXT_AM);
		strcpy_P(tbuf, PSTR("FM"));
		LCD3300_DrawText(tbuf, 8*0, 12*2, COLOR_BG, 0b10010001);
		strcpy_P(tbuf, PSTR("kHz"));
		LCD3300_DrawText(tbuf, 8*4+64, 12*3, COLOR_BG, COLOR_TEXT_AM);
		uint16_t freq = BK1088_GetCurrentFreq();
		// freq -> text
		{
			tbuf[0] = tbuf[1] = tbuf[2] = tbuf[3] = '0';
			tbuf[4] = '\0';
			while(freq >= 1000){
				tbuf[0] ++;
				freq -= 1000;
			}
			if(tbuf[0] == '0') tbuf[0] = '9'+2; // ' '
			while(freq >= 100){
				tbuf[1] ++;
				freq -= 100;
			}
			while(freq >= 10){
				tbuf[2] ++;
				freq -= 10;
			}
			tbuf[3] += freq;
		}
		// show freq
		LCD3300_DrawBigText(tbuf, 6*3, 12*2, COLOR_BG, COLOR_TEXT_AM);
	}else if(g_radioSettings.mode == CHANNEL_SW){
		strcpy_P(tbuf, PSTR("SW"));
		LCD3300_DrawText(tbuf, 8*0, 12*3, COLOR_BG, COLOR_TEXT);
		strcpy_P(tbuf, PSTR("FM"));
		LCD3300_DrawText(tbuf, 8*0, 12*2, COLOR_BG, 0b10010001);
		strcpy_P(tbuf, PSTR("kHz"));
		LCD3300_DrawText(tbuf, 8*5+64, 12*3, COLOR_BG, COLOR_TEXT);
		uint16_t freq = BK1088_GetCurrentFreq();
		// freq -> text
		{
			tbuf[0] = tbuf[1] = tbuf[2] = tbuf[3] = tbuf[4] = '0';
			tbuf[5] = '\0';
			while(freq >= 10000){
				tbuf[0] ++;
				freq -= 10000;
			}
			if(tbuf[0] == '0') tbuf[0] = '9'+2; // ' '
			while(freq >= 1000){
				tbuf[1] ++;
				freq -= 1000;
			}
			while(freq >= 100){
				tbuf[2] ++;
				freq -= 100;
			}
			while(freq >= 10){
				tbuf[3] ++;
				freq -= 10;
			}
			tbuf[4] += freq;
		}
		// show freq
		LCD3300_DrawBigText(tbuf, 6*3, 12*2, COLOR_BG, COLOR_TEXT);
	}else if(g_radioSettings.mode == CHANNEL_LW){
		strcpy_P(tbuf, PSTR("LW"));
		LCD3300_DrawText(tbuf, 8*0, 12*2, COLOR_BG, COLOR_TEXT);
		strcpy_P(tbuf, PSTR("FM"));
		LCD3300_DrawText(tbuf, 8*0, 12*3, COLOR_BG, 0b10010001);
		strcpy_P(tbuf, PSTR("kHz"));
		LCD3300_DrawText(tbuf, 8*4+64, 12*3, COLOR_BG, COLOR_TEXT);
		uint16_t freq = BK1088_GetCurrentFreq();
		// freq -> text
		{
			tbuf[0] = tbuf[1] = tbuf[2] = tbuf[3] = '0';
			tbuf[4] = '\0';
			while(freq >= 1000){
				tbuf[0] ++;
				freq -= 1000;
			}
			if(tbuf[0] == '0') tbuf[0] = '9'+2; // ' '
			while(freq >= 100){
				tbuf[1] ++;
				freq -= 100;
			}
			while(freq >= 10){
				tbuf[2] ++;
				freq -= 10;
			}
			tbuf[3] += freq;
		}
		// show freq
		LCD3300_DrawBigText(tbuf, 6*3, 12*2, COLOR_BG, COLOR_TEXT);
	}
	
	// Broadcast description
	LCD3300_DrawText(g_channelDescription, 0*6 ,12*5, 0b10010001, 0b11111111);
	strcpy_P(tbuf, PSTR("                        "));
	LCD3300_DrawText(tbuf, strlen(g_channelDescription)*6 ,12*5, 0b10010001, 0b11111111);
	
	
	// Show Mode

//	strcpy_P(tbuf, PSTR(" Channel"));
//	LCD3300_DrawText(tbuf, 6*0, 128-12-1, 0x00, 0b11110100);
	strcpy_P(tbuf, PSTR(" Mode: "));
	LCD3300_DrawText(tbuf, 6*0, 128-12-1, 0x00, 0xFF);
	tuning = (g_radioSettings.config >> 2) & 0x03; // tuning mode
	if(tuning  == 0){
		strcpy_P(tbuf, PSTR("AUTO    "));
		LCD3300_DrawText(tbuf, 6*7, 128-12-1, 0x00, 0b11100000);
	}
	else if(tuning  == 1){
		strcpy_P(tbuf, PSTR("PRESET  "));
		LCD3300_DrawText(tbuf, 6*7, 128-12-1, 0x00, 0b00011100);
	}
	else if(tuning  == 2){
		strcpy_P(tbuf, PSTR("DIRECT  "));
		LCD3300_DrawText(tbuf, 6*7, 128-12-1, 0x00, 0b00001011);
	}
}

/*static uint8_t strlen_usr(char* a){
	uint8_t ret = 0;
	while(a[ret] != '\0') ret++;
	return ret;
}*/


// Switch check routine
static uint8_t CheckSW(void){
	uint8_t i;
	uint8_t sw = g_swState;
	uint16_t a;
	g_swState = 0;
	
	if((g_backlight==0) && sw){
		g_swState = 0;
		LCD3300_SleepOut();
		return 0;
	}

	// backlight brightness
	if(g_timeAfterSwitch > g_blOffBySec){
		if(g_backlight) {
			LCD3300_SleepIn();
		}
		g_backlight = 0;
		CBI(PORTD, PD5);
	}else if(g_timeAfterSwitch > g_blWeakBySec){
		g_backlight = 2;
		SBI(PORTD, PD5);
	}else{
		g_backlight = 1;
		SBI(PORTD, PD5);
	}
	
	uint8_t tuning = (g_radioSettings.config >> 2) & 0x03; // tuning mode
	
	// main part
	if(sw == SW1){
		uint8_t b = 0;
		if(g_radioSettings.mode == CHANNEL_FM){
			g_radioSettings.mode = CHANNEL_MW;
			b = 1;
			g_radioSettings.prevChannel[0] = BK1088_GetCurrentChannel();
		}else if(g_radioSettings.mode == CHANNEL_MW){
			g_radioSettings.mode = CHANNEL_SW;
			b = 2;
			g_radioSettings.prevChannel[1] = BK1088_GetCurrentChannel();
		}else if(g_radioSettings.mode == CHANNEL_SW){
			g_radioSettings.mode = CHANNEL_FM;
			b = 0;
			g_radioSettings.prevChannel[2] = BK1088_GetCurrentChannel();
		}
		BK1088_SetMode(g_radioSettings.mode);
		LCD3300_DrawRect(0,0,128,128, COLOR_BG); // Clear all display
		
		// change ANT switch (AM,FM / SW)
		ChangeANTSwitch(0);
		
		// try to tune defined channel
		if(g_radioSettings.prevChannel[b]){
			BK1088_SetChannel(g_radioSettings.prevChannel[b]);
		}else{
			uint8_t ret = PMGetNextPreset(g_radioSettings.mode, 0, &a, g_channelDescription, 1);
			if(ret == 0) BK1088_SetChannel(a);
		}
	}else if(sw == SW2){
		if(tuning == 0){ // Auto -> Preset
			// stop tuning process
			a = BK1088_GetCurrentChannel();
		//	uint8_t ret = PMGetNextPreset(g_radioSettings.mode, a, &a, g_channelDescription, 1);
		//	if(ret == 0) BK1088_SetChannel(a);
		}
		tuning++;
		if(tuning > 2) tuning = 0; // Direct -> Auto
		
		g_radioSettings.config = (g_radioSettings.config & 0xF3) | (tuning << 2);
	}else if(sw == SW3){ // FWD
		if(tuning == 0){// auto
			strcpy_P(tbuf, PSTR("Searching...       "));
			LCD3300_DrawText(tbuf, 0, 12*1, 0x00, 0b11100000);
			BK1088_SeekNext2(1);
			strcpy_P(tbuf, PSTR("            "));
			LCD3300_DrawText(tbuf, 0, 12*1, 0x00, 0b11100000);
			g_timeAfterSwitch = 0;
		}
		else if(tuning  == 1){ // preset
			uint8_t ret;
			a = BK1088_GetCurrentChannel();
			ret = PMGetNextPreset(g_radioSettings.mode, a+1, &a, g_channelDescription, 1);
			if(ret == 0) BK1088_SetChannel(a);
			else{
				ret = PMGetNextPreset(g_radioSettings.mode, 0, &a, g_channelDescription, 1);
				if(ret == 0) BK1088_SetChannel(a);
			}
		}
		else if(tuning  == 2){ // direct
			a = BK1088_GetCurrentChannel();
			
			uint16_t maxf = 0;
			if(g_radioSettings.mode == CHANNEL_MW){
				maxf = CHANNEL_MW_MAX-1;
			}else if(g_radioSettings.mode == CHANNEL_SW){
				maxf = CHANNEL_SW_MAX-1;
			}else if(g_radioSettings.mode == CHANNEL_LW){
				maxf = CHANNEL_LW_MAX-1;
			}else{
				maxf = CHANNEL_FM_MAX-1;
			}
			if(a >= maxf) a=0;
			else a++;
			BK1088_SetChannel(a);
		}
	}else if(sw == SW4){ // REV
		if(tuning == 0){// auto
			strcpy_P(tbuf, PSTR("Searching...       "));
			LCD3300_DrawText(tbuf, 0, 12*1, 0x00, 0b11100000);
			BK1088_SeekNext2(0);
			strcpy_P(tbuf, PSTR("            "));
			LCD3300_DrawText(tbuf, 0, 12*1, 0x00, 0b11100000);
			g_timeAfterSwitch = 0;
		}
		else if(tuning  == 1){ // preset
			uint8_t ret;
			a = BK1088_GetCurrentChannel();
			ret = PMGetNextPreset(g_radioSettings.mode, a, &a, g_channelDescription, 0);
			if(ret == 0) BK1088_SetChannel(a);
			else{
				ret = PMGetNextPreset(g_radioSettings.mode, 0xFFF0, &a, g_channelDescription, 0);
				if(ret == 0) BK1088_SetChannel(a);
			}
		}
		else if(tuning  == 2){ // direct
			a = BK1088_GetCurrentChannel();
			if(a == 0) {
				if(g_radioSettings.mode == CHANNEL_MW){
					a = CHANNEL_MW_MAX;
				}else if(g_radioSettings.mode == CHANNEL_SW){
					a = CHANNEL_SW_MAX;
				}else if(g_radioSettings.mode == CHANNEL_LW){
					a = CHANNEL_LW_MAX;
				}else{
					a = CHANNEL_FM_MAX;
				}
			}
			BK1088_SetChannel(a-1);
		}
	}
	else if(sw == (SW3 | SW_PUSHING)){ // Volume +
			if(g_radioSettings.vol < 31) g_radioSettings.vol ++;
			BK1088_SetVolume();
	}
	else if(sw == (SW4 | SW_PUSHING)){ // Volume +
			if(g_radioSettings.vol > 0) g_radioSettings.vol --;
			BK1088_SetVolume();
	}
	else if(sw == (SW1 | SW_LONG)) { // Power off
		strcpy_P(tbuf, PSTR("POWER OFF.       "));
		LCD3300_DrawText(tbuf, 0, 12*1, 0x00, 0b11100000);
		g_radioSettings.channel = BK1088_GetCurrentChannel();
		SaveEEP();
		//
		for(i=0; i<100; i++) _delay_ms(5);
		LCD3300_PowOff();
		while(!(PINB & 0b00000100)); // wait if MENU_SW is pushed
		CBI(PORTC, 0);
		while(1);
	}
	else if(sw == (SW2 | SW_LONG)){ // debug
//		while(g_swState == 0);
//		g_swState = 0;
		
		static uint8_t temp;
		temp++;
		
		ChangeANTSwitch(temp);
		
		if(temp & 0x01){
			strcpy_P(tbuf, PSTR("ANT: EXTERNAL"));
			LCD3300_DrawText(tbuf, 0, 12*4, 0x00, 0b11100000);
			BK1088_Send(24, 0xB41C);
		}else{
			strcpy_P(tbuf, PSTR("ANT: INTERNAL"));
			LCD3300_DrawText(tbuf, 0, 12*4, 0x00, 0b11100000);
			BK1088_Send(24, 0x341C);
		}
		if(temp & 0x02){
			CBI(PORTC, PC4); // Preamp OFF
		}else{
			SBI(PORTC, PC4); // Preamp ON
		}
		a = BK1088_GetCurrentChannel();
		BK1088_SetChannel(a);
	}
	
	if(sw){
		// update current channel description
		a = BK1088_GetCurrentChannel();
		PMGetPresetDescription(g_radioSettings.mode, a, g_channelDescription);
	}
	
	if(sw) return sw;
	else return 0;
}

int main(void){
	uint8_t i;
	// Load initial value (Radio)
	g_radioSettings.mode = CHANNEL_FM; //FM
	g_radioSettings.config = 0x01;
	g_radioSettings.channel = 0;
	g_radioSettings.vol = 23;
	g_radioSettings.SNR_thre_FM  = 6;
	g_radioSettings.RSSI_thre_FM = 27;
	g_radioSettings.SNR_thre_AM  = 21;
	g_radioSettings.RSSI_thre_AM = 16;

	InitPort();
	// if SW4 is pushed, skip loading EEPROM data
	if(SW_PORT & 0b00100000) LoadEEP();
	
	// 
	if(Init()){
		// Show Error Msg.	
		strcpy_P(tbuf, PSTR("Error: BK1088 init"));
		LCD3300_DrawText(tbuf, 0, 12*1, 0x00, 0b11100000);
		for(i=0; i<100; i++) _delay_ms(5);
		// Power off.
		return -1;
	}
	ChangeANTSwitch(0);
	
	// wait while sw is pushing
	LCD3300_DrawRect(0,0,128,128, COLOR_BG);
	strcpy_P(tbuf, PSTR("Release switch(s)."));
	LCD3300_DrawText(tbuf, 0, 12*1, COLOR_BG, COLOR_TEXT);
	while( (SW_PORT & SW_PORT_MASK) != SW_PORT_MASK){
		for(i=0; i<100; i++) _delay_ms(1);
	}
	g_swState = 0;
	
	LCD3300_DrawRect(0,0,128,128, COLOR_BG);
//	LCD3300_DrawRect(0,64,128,128, 0x0F);
	
	BK1088_UpdateStatus();
	BK1088_SetVolume();
	BK1088_SetMode(g_radioSettings.mode);
	BK1088_SetChannel(g_radioSettings.channel);
	PMGetPresetDescription(g_radioSettings.mode, g_radioSettings.channel, g_channelDescription);
	ShowRadioDisplay();
	
	// const char* text, uint8_t x0, uint8_t y0, uint8_t bgColor, uint8_t textColor
	while(1){
		
		// check switch
		uint8_t a = CheckSW();
		
		if(a) ShowRadioDisplay();
		
		if(g_backlight){
			// update display
//			ShowRadioDisplay();
			
			// show overwrap display
			ShowOverwrapDisplay();
			
			for(i=0; i<100; i++) _delay_ms(2);
		//	if(g_radioSettings.mode == CHANNEL_FM) {
				BK1088_UpdateStatus();
		//	}
		}else{
			for(i=0; i<100; i++) _delay_ms(1);
		}
	}

	return 0;
}
