<?php

/**
 * sfSiteStructure sitemap,topic path,breadcrumbs
 * Singleton
 *
 * @package    symfony
 * @subpackage plugin
 * @author     flyfront <flyfront@gmail.com>
 * @version    SVN: $Id: actions.class.php 19 2008-05-01 05:57:57Z flyfront $
 */
class sfSiteStructure
{
    static protected $instance = null;
    
    /**
     * ルートノード(sfStructureNode)の配列
     * ※structure.ymlでは複数のルートノードが設定可能
     */
    protected $structure_tree;
    
    /**
     * ルーティング名=>ノードインスタンス(sfStructureNode)の連想配列
     */
    protected $nodes;
    
    /**
     * $nodeと同じ用途だが、動的要素のルーティング名+パラメータとノードの組を保存
     * 
     * ex. @routing_name?id=4&type=update
     * array(
     *   'routing_name' => array(
     *     'params' => array( 'id' => 4, 'type'=> 'updated' ),
     *     'node' => sfStructureNode(routingには動的URL)
     *   );
     * );
     */
    protected $dynamic_nodes;
    
    /**
     * 各ノード(ルーティング)間親子関係の連想配列
     * 子ルーティング名 => 親ルーティング名
     */
    protected $structure_relations;
    
    /**
     * ルーティング名=>ページタイトルの連想配列
     */
    protected $page_titles;
    
    /**
     * 以下のプロパティは
     * 上記のサイト構造データを生成するための
     * 各build系メソッドを実行する際に使用します。
     */
    
    /**
     * app/config/structure.yml のデータ配列
     */
    protected $structure;    
    
    /**
     * app/config/view.yml のデータ配列
     */
    protected $view_config;
    
    /**
     * app/config/routings.yml のデータ配列
     */
    protected $routings;
    
    protected function __construct()
    {
    }
    
    protected function initialize()
    {
        $this->routings = sfYaml::load(sfConfig::get('sf_app_config_dir').'/routing.yml');
        $this->view_config = sfYaml::load(sfConfig::get('sf_app_config_dir').'/view.yml');
        $this->structure = $this->getStructureConfig();
        
        //set $this->page_titles
        $this->buildPageTitles();
        //set $this->structure_tree
        $this->buildStructureTree();
    }
    
    /**
     * @return sfSiteStructure
     */
    static public function getInstance()
    {
        if (!isset(self::$instance))
        {
            $class = __CLASS__;
            self::$instance = new $class();
            self::$instance->initialize();
        }
    
        return self::$instance;
    }
    
    static public function removeInstance()
    {
        self::$instance = null;
    }
    
    /**
     * app/config/structure.yml を読み込んで、データ配列を返す
     */
    protected function getStructureConfig()
    {
        if ($this->structure) {
            return $this->structure;
        }
        $structure = sfYaml::load(sfConfig::get('sf_app_config_dir').'/structure.yml');
        
        /**
         * 一層目にあるプロパティ要素(_hoge)を除去しておく
         * 仕様としては、一層目にプロパティ要素はこないはずだが、念のため
         */
            
        unset($structure['_structure_info']);
        if (0 === count($structure)){
            $structure = null;
        }
        if (!$structure && !is_array($structure)){
            throw new sfSiteStructureException('structure.ymlからサイト構造が取得できませんでした。');
        }
        $this->structure = $structure;
        return $this->structure;
    }
    
    /**
     * app/config/view.yml を読み込んで、データ配列を返す
     */
    protected function getAppViewConfig()
    {
        if (!$this->view_config) {
            $this->view_config = sfYaml::load(sfConfig::get('sf_app_config_dir').'/view.yml');
        }
        return $this->view_config;
    }
    
    /**
     * app/config/routings.yml を読み込んで、データ配列を返す
     */
    protected function getRoutingsConfig()
    {
        if (!$this->routings) {
            $this->routings = sfYaml::load(sfConfig::get('sf_app_config_dir').'/routing.yml');
        }
        return $this->routings;
    }
    
    public function getStructureTree()
    {
        if (!$this->structure_tree){
            $this->buildStructureTree();
        }
        return $this->structure_tree;
    }
    
    /**
     * $this->structure をもとに
     * sfStructureNodeクラスの階層構造を構築する
     */
    public function buildStructureTree()
    {
        //buildStructureTreeRecursive内部で値が設定されるので空にする
        $this->structure_relations = array();
        $this->nodes = array();
        
        $this->structure_tree = $this->buildStructureTreeRecursive($this->getStructureConfig(), null);
    }
    
    /**
     * buildStructureTree()内部で使う再帰関数
     * 
     * @param $parent_node sfStructureNode
     */
    protected function buildStructureTreeRecursive($parent_structures, $parent_node)
    {
        $parent_nodes = array();
        foreach ($parent_structures as $child_name => $child_structure)
        {
            /**
             * 初期値
             */
            if (isset($this->page_titles[$child_name])) {
            	$title = $this->page_titles[$child_name];
            }else {
            	$title = null;
            }
            $is_nopage = false; //実体のあるページ
            $is_dynamic = false;    //動的(1ルーティング複数ページ)では"無い"
            if (strpos($child_name, '?')){
                list($child_name, $child_params) = sfContext::getInstance()->getController()->convertUrlStringToParameters('@'.$child_name);
            }else{
                $child_params = array();
            }
            
            /**
             * sfSiteStructure用の指定取得
             */
            if (isset($child_structure['_structure_info'])) {
            	$info = $child_structure['_structure_info'];
            	if (isset($info['title'])){
                    $title = $info['title'];
                }
                if (isset($info['nopage']) && $info['nopage']){
                    $is_nopage = true;
                    if (!$title) {
                        throw new sfSiteStructureException('structure.ymlのルーティング名".$child_name."で_nopage指定をした場合は、必ず_titleを設定してください。');
                    }
                }
                if (isset($info['dynamic'])) {
                    $is_dynamic = true;
                }
                unset($child_structure['_structure_info']);
                if (0 === count($child_structure)){
                    //info以外ないでの、ここが終端
                    $child_structure = null;
                }
            }
            
            /**
             * 自ノードを作成、静的(1ルーティングに1ページ)なら1ノード
             * 動的フラグ(動的データ取得メソッドの指定)がある場合は
             * 指定メソッドから得られるデータを元に、複数のノードを作成する。
             */
            $current_nodes = array();
            if (!$is_dynamic) {
                $current_node = new sfStructureNode();
                $current_node->initialize($child_name, $title, $is_nopage, $child_params);
                $current_nodes[] = $current_node;
            }else {
                $dynamic_pages = call_user_func($info['dynamic'], &$parent_node);
                foreach ($dynamic_pages as $dynamic_page) {
                    $dynamic_node = new sfStructureNode();
                    $dynamic_node->initialize($child_name, $dynamic_page['title'], $is_nopage, $dynamic_page['params'], array());
                    $current_nodes[] = $dynamic_node;
                }
            }
            
            /**
             * 現ルーティングノードを親/子ノードを関連づける
             * ※子ノードの作成(再帰)も行う
             */
            foreach ($current_nodes as $current_node) {
                /**
                 * $parent_node空=ルートノード
                 * ルートノードはだれの子でもないため、relationは無視
                 */
                if ($parent_node){
                    $current_node->setParentRouting($parent_node->getRouting(), $parent_node->getParams());
                }
                
                /**
                 * 子があるなら子ノードを作る(再帰処理)
                 */
                $child_nodes = array();
                if (is_array($child_structure)){
                    $child_nodes = $this->buildStructureTreeRecursive($child_structure, $current_node);
                }
                $current_node->setChildNodes($child_nodes);
                $parent_nodes[] = $current_node;
                $this->nodes[$current_node->getRouting()][] = array(
                    'node' => $current_node,
                    'params' => $current_node->getParams()
                );
            }
        }
        return $parent_nodes;
    }
    
    public function getAllNodes()
    {
        if (!$this->nodes){
            $this->buildStructureTree();
        }
        return $this->nodes;
    }
    
    /**
     * @param string ノードインスタンスを取得したいルーティング名
     * @param array ノードインスタンスを取得する際の追加パラメータ
     * @return sfStructureNode ノードインスタンス
     */
    public function getNodeByRouting($routing, $params = array())
    {
        $routing = $this->trimCommercialAtMark($routing);
        if (!isset($this->nodes[$routing])){
            return null;
        }
        if(is_array($this->nodes[$routing]) && is_array($params)) {
            foreach ($this->nodes[$routing] as $nodes) {
                /**
                 * ルーティング指定に無いパラメータがRequestに来る場合もあり
                 * $paramsのパラメータ数の方が多いこともある
                 * 
                 * array_diff(_assoc)関数は
                 * 第一引数配列にあるものでしかチェックしない仕様を利用した
                 */
                if (array() === array_diff_assoc($nodes['params'],$params)){
                    return $nodes['node'];
            	}
            }
        }
        return null;
    }
    
    public function getPageTitles()
    {
        if (!$this->page_titles) {
        	$this->buildPageTitles();
        }
        return $this->page_titles;
    }
    
    protected function buildPageTitles()
    {
        $routings = $this->getRoutingsConfig();
        $page_titles = array();
        foreach ($routings as $routing_name => $routing_value) {
        	$page_titles[$routing_name] = $this->getPageTitleByRoutingName($routing_name, $routing_value);
        }
        $this->page_titles = $page_titles;
    }
    
    /**
     * app全体と各モジュールのview.ymlからページタイトルを取得する
     * なお、タイトル取得アルゴリズムはsymfony本体と同じにした
     * 
     * @see sfViewConfigHandler
     */
    public function getPageTitleByRoutingName($routing_name, $routing_value)
    {
        /**
         * ルーティング名からmodule/actionを取得
         */
        if (!isset($routing_value['param'])){
            throw new sfSiteStructureException('structure.ymlで指定したルーティング名"'.$routing_name.'"がroutings.ymlに存在しません');
        }
        $routing_param  = $routing_value['param'];
        
        /**
         * moduleのview設定の取得
         */
        $module_view_path = sfConfig::get('sf_app_module_dir').'/'.$routing_param['module'].'/'.sfConfig::get('sf_app_module_config_dir_name').'/view.yml';
        if (is_readable($module_view_path)){
            $module_view = sfYaml::load($module_view_path);
        }else {
        	$module_view = array();
        }
        
        /**
         * app全体のviewとmoduleのviewをマージ
         */
        $view_config = sfToolkit::arrayDeepMerge($this->view_config, $module_view);
        $view_config['all'] = sfToolkit::arrayDeepMerge(
          isset($view_config['default']) && is_array($view_config['default']) ? $view_config['default'] : array(),
          isset($view_config['all']) && is_array($view_config['all']) ? $view_config['all'] : array()
        );
        unset($view_config['default']);
        
        /**
         * その中から[action名]Success時のタイトルを取得、なければallを使う
         */
        $view_config = (isset($view_config[$routing_param['action'].'Success'])) ? $view_config[$routing_param['action'].'Success']: $view_config['all'];
        if (!isset($view_config['metas']['title'])) {
            return null;
        }
        return $view_config['metas']['title'];
    }
    
    /**
     * ルーティング名の先頭の@は無視したいので、削除する
     * 
     * @param string ルーティング名(@有り/無し)
     * @return string @を除いたルーティング名
     */
    public function trimCommercialAtMark($routing)
    {
    	if ('@' === substr($routing, 0, 1)){
    	    $routing = substr($routing, 1);
    	}
    	return $routing;
    }
}