#region Using Statements
using System;
using System.Collections.Generic;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Storage;
#endregion

namespace Chemics
{
    public interface IGraphicsContent
    {
        void LoadGraphicsContent(bool loadAllContent);
    }

    interface IGraphicsContentManager
    {
        void Add(IGraphicsContent content);
    }

    /// <summary>
    /// This is the main type for your game
    /// </summary>
    public class Game1 : Microsoft.Xna.Framework.Game, IGraphicsContentManager
    {
        #region Variables

        GraphicsDeviceManager graphics;
        ContentManager content;
        BasicEffect effect;

        SphericalHarmonicsManager modelManager = new SphericalHarmonicsManager();
        private bool frameOn;
        private List<IGraphicsContent> graphicsContents = new List<IGraphicsContent>();

        AxisXYZModel axis = new AxisXYZModel();
        TextRenderer textRenderer = new TextRenderer();


        private bool lastTabPressed;
        private bool lastMouseButtonPressed;
        private Vector2 previousMousePosition;

        #endregion

        #region Initialization
        public Game1()
        {
            graphics = new GraphicsDeviceManager(this);
            content = new ContentManager(Services);
            Services.AddService(typeof(ContentManager), content);
            Services.AddService(typeof(IGraphicsContentManager), this);
        }


        /// <summary>
        /// Allows the game to perform any initialization it needs to before starting to run.
        /// This is where it can query for any required services and load any non-graphic
        /// related content.  Calling base.Initialize will enumerate through any components
        /// and initialize them as well.
        /// </summary>
        protected override void Initialize()
        {
            modelManager.Services = Services;
            modelManager.Initialize();

            axis.Services = Services;
            axis.Initialize();

            textRenderer.Services = Services;
            textRenderer.Initialize();

            base.Initialize();
        }





        /// <summary>
        /// Load your graphics content.  If loadAllContent is true, you should
        /// load content from both ResourceManagementMode pools.  Otherwise, just
        /// load ResourceManagementMode.Manual content.
        /// </summary>
        /// <param name="loadAllContent">Which type of content to load.</param>
        protected override void LoadGraphicsContent(bool loadAllContent)
        {
            if (loadAllContent)
            {
                effect = new BasicEffect(graphics.GraphicsDevice, null);
                effect.View = Matrix.CreateLookAt(
                    new Vector3(0, 0, 3),
                    new Vector3(),
                    new Vector3(0, 1, 0)
                    );
                effect.Projection = Matrix.CreatePerspectiveFieldOfView(
                    MathHelper.ToRadians(45),
                    AspectRatio,
                    1,
                    100
                    );

                Services.RemoveService(typeof(BasicEffect));
                Services.AddService(typeof(BasicEffect), effect);
            }

            foreach (IGraphicsContent content in graphicsContents)
            {
                content.LoadGraphicsContent(loadAllContent);
            }

            // TODO: Load any ResourceManagementMode.Manual content
        }

        float AspectRatio
        {
            get
            {
                return Window.ClientBounds.Width / (float)Window.ClientBounds.Height;
            }
        }
#endregion


        public void Add(IGraphicsContent content)
        {
            graphicsContents.Add(content);

            if (graphics.GraphicsDevice != null)
            {
                content.LoadGraphicsContent(true);
            }
        }

        /// <summary>
        /// Unload your graphics content.  If unloadAllContent is true, you should
        /// unload content from both ResourceManagementMode pools.  Otherwise, just
        /// unload ResourceManagementMode.Manual content.  Manual content will get
        /// Disposed by the GraphicsDevice during a Reset.
        /// </summary>
        /// <param name="unloadAllContent">Which type of content to unload.</param>
        protected override void UnloadGraphicsContent(bool unloadAllContent)
        {
            if (unloadAllContent)
            {
                // TODO: Unload any ResourceManagementMode.Automatic content
                content.Unload();
            }

            // TODO: Unload any ResourceManagementMode.Manual content
        }

        #region Update
        /// <summary>
        /// Allows the game to run logic such as updating the world,
        /// checking for collisions, gathering input and playing audio.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Update(GameTime gameTime)
        {
            MouseState mouseState = Mouse.GetState();

            rotateModel(mouseState);

            


            KeyboardState keyState = Keyboard.GetState();
            frameOn = keyState.IsKeyDown(Keys.Space);

            changeModel(keyState);
            rotateModel(keyState);

            base.Update(gameTime);
        }

        private void rotateModel(MouseState mouseState)
        {
            if (HandednessUtility.IsClicking(mouseState))
            {
                if (lastMouseButtonPressed)
                {
                    effect.World *= Matrix.CreateRotationY(
                        0.01f * (mouseState.X - previousMousePosition.X)
                        );
                    effect.World *= Matrix.CreateRotationX(
                        0.01f * (mouseState.Y - previousMousePosition.Y)
                        );

                }

                previousMousePosition.X = mouseState.X;
                previousMousePosition.Y = mouseState.Y;

            }

            lastMouseButtonPressed = HandednessUtility.IsClicking(mouseState);
        }

        private void rotateModel(KeyboardState keyState)
        {
            if (keyState.IsKeyDown(Keys.Up))
            {
                effect.World *= Matrix.CreateRotationX(-0.05f);
            }
            if (keyState.IsKeyDown(Keys.Down))
            {
                effect.World *= Matrix.CreateRotationX(0.05f);
            }
            if (keyState.IsKeyDown(Keys.Left))
            {
                effect.World *= Matrix.CreateRotationY(-0.05f);
            }
            if (keyState.IsKeyDown(Keys.Right))
            {
                effect.World *= Matrix.CreateRotationY(0.05f);
            }
        }

        private void changeModel(KeyboardState keyState)
        {
            if (!keyState.IsKeyDown(Keys.Tab) && lastTabPressed)
            {
                if (keyState.IsKeyDown(Keys.RightShift))
                { modelManager.Previous(); }
                else
                { modelManager.Next(); }
            }

            lastTabPressed = keyState.IsKeyDown(Keys.Tab);
            modelManager.IsSquared = keyState.IsKeyDown(Keys.A);
        }
        #endregion


        #region Draw

        /// <summary>
        /// This is called when the game should draw itself.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        protected override void Draw(GameTime gameTime)
        {
            graphics.GraphicsDevice.Clear(Color.CornflowerBlue);


            effect.Begin();

            effect.EnableDefaultLighting();
            effect.VertexColorEnabled = true;

            axis.Draw();


            effect.Alpha = 0.5f;
            graphics.GraphicsDevice.RenderState.DepthBufferWriteEnable = false;
            graphics.GraphicsDevice.RenderState.AlphaBlendEnable = true;
            graphics.GraphicsDevice.RenderState.DestinationBlend = Blend.InverseSourceAlpha;
            graphics.GraphicsDevice.RenderState.SourceBlend = Blend.SourceAlpha;

            drawSphericalHarmonicsModel();

            graphics.GraphicsDevice.RenderState.AlphaBlendEnable = false;
            graphics.GraphicsDevice.RenderState.DepthBufferWriteEnable = true;


            if (frameOn)
            {
                drawFrame();
            }


            drawCurrentStates();

            effect.End();

            base.Draw(gameTime);
        }

        private void drawCurrentStates()
        {
            textRenderer.Begin();
            textRenderer.DrawLine(modelManager.SelectedModel.Parameters.ToString());
            textRenderer.DrawLine("IsSquared : " + modelManager.IsSquared.ToString());

            textRenderer.ChangePosition(new Vector2(0, 300));
            textRenderer.DrawLine("Usage");
            textRenderer.DrawLine("Change Model : Tab");
            textRenderer.DrawLine("Square : A");
            textRenderer.DrawLine("Wire Frame : Space");
            textRenderer.DrawLine("Rotate Model : Drag");
            textRenderer.DrawLine("or Up, Down, Left or Right key");
            textRenderer.End();
        }


        private void drawSphericalHarmonicsModel()
        {
            foreach (EffectPass pass in effect.CurrentTechnique.Passes)
            {
                pass.Begin();

                modelManager.SelectedModel.Draw();

                pass.End();
            }
        }

        private void drawFrame()
        {
            effect.VertexColorEnabled = false;
            graphics.GraphicsDevice.RenderState.FillMode = FillMode.WireFrame;
            graphics.GraphicsDevice.RenderState.DepthBias = -0.0001f;

            drawSphericalHarmonicsModel();

            graphics.GraphicsDevice.RenderState.DepthBias = 0;
            graphics.GraphicsDevice.RenderState.FillMode = FillMode.Solid;
            effect.VertexColorEnabled = true;
        }
        #endregion
    }
}
