using System;
using System.Collections.Generic;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;

namespace Chemics
{
    class SphericalHarmonicsModel:ColorModel
    {
        private SphericalHarmonics sphericalHarmonics = new SphericalHarmonics();
        private bool isSquared;

        const int separation = 30;
        const double difference = Math.PI / separation;

        #region Properties

        public SphericalHarmonicsParameters Parameters
        {
            get { return sphericalHarmonics.Parameters; }
        }

        public bool IsSquared
        {
            get { return isSquared; }
        }

        #endregion

        public SphericalHarmonicsModel(
            int azimuthalQuantumNumber,
            int magneticQuantumNumber,
            bool isSquared)
        {
            sphericalHarmonics.Parameters.AzimuthalQuantumNumber = azimuthalQuantumNumber;
            sphericalHarmonics.Parameters.MagneticQuantumNumber = magneticQuantumNumber;
            this.isSquared = isSquared;
        }


        public void Initialize()
        {

            for (int thetaIndex = 0; thetaIndex < separation; thetaIndex++)
            {
                double theta = Math.PI * thetaIndex / separation;

                for (int phiIndex = 0; phiIndex < separation * 2; phiIndex++)
                {
                    double phi = Math.PI * phiIndex / (separation);

                    VertexPositionNormalColor v00 = getVertex(theta, phi);
                    VertexPositionNormalColor v10 = getVertex(theta + difference, phi);
                    VertexPositionNormalColor v01 = getVertex(theta, phi + difference);
                    VertexPositionNormalColor v11 = getVertex(theta + difference, phi + difference);
                    Vertices.Add(v00);
                    Vertices.Add(v11);
                    Vertices.Add(v10);

                    Vertices.Add(v00);
                    Vertices.Add(v01);
                    Vertices.Add(v11);
                }
            }


            GetService<IGraphicsContentManager>().Add(this);
        }

        VertexPositionNormalColor getVertex(double theta, double phi)
        {
            VertexPositionNormalColor result = new VertexPositionNormalColor();
            Complex value = sphericalHarmonics.GetValue(theta, phi);

            if (isSquared)
            { value = value.LengthSquared(); }

            if (value.R > 0)
            {
                result.Position = getSphericalPosition(theta, phi, value.R);
                result.Color = Color.Red;
                result.Normal = getNormal(theta, phi, value.R);
            }
            else
            {
                result.Position = getSphericalPosition(theta, phi, -value.R);
                result.Color = Color.Blue;
                result.Normal = getNormal(theta, phi, -value.R);
            }
            return result;
        }

        Vector3 getNormal(double theta, double phi, double radius)
        {
            Vector3 v1 = getSphericalPosition(theta, phi, radius);
            Vector3 v2 = getSphericalPosition(theta + difference/2, phi, radius);
            Vector3 v3 = getSphericalPosition(theta + difference/2, phi + difference/2, radius);
            Vector3 cross = Vector3.Cross(v2 - v1, v3 - v1);

            return Vector3.Normalize(cross);
        }

        Vector3 getSphericalPosition(double theta, double phi, double radius)
        {
            Vector3 result = new Vector3();
            double zRadius = Math.Sin(theta) * radius;
            result.X = (float)(Math.Cos(phi) * zRadius);
            result.Y = (float)(Math.Sin(phi) * zRadius);
            result.Z = (float)(Math.Cos(theta) * radius);
            return result;
        }

    }
}
