using System;
using System.Collections.Generic;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Content;


namespace MyFireballTest
{
    public class FireballRenderer : DrawableGameComponent
    {
        #region Variables
        ContentManager content;

        Effect effect;
        FireVertex[] vertices;
        VertexDeclaration vertexDeclaration;

        static Random random = new Random();
        const float particleSpeed = 1f;
        const float particleSize = 1.8f;
        float duration = 2;

        List<Fireball> fireballs = new List<Fireball>();

        Matrix view = Matrix.Identity;
        Matrix projection = Matrix.Identity;
        #endregion

        #region Initialization
        public FireballRenderer(Game game)
            : base(game)
        {
            content = new ContentManager(game.Services);
        }


        public override void Initialize()
        {
            vertices = new FireVertex[500];

            for (int i = 0; i < vertices.Length; i++)
            {
                vertices[i].Position = RandomNormalizedVector3() * (float)random.NextDouble();
                vertices[i].Size = particleSize;
                vertices[i].Velocity = RandomNormalizedVector3() *particleSpeed;
                vertices[i].StartTime = -vertices[i].Position.Length() * 2;
                vertices[i].Random = RandomVector4();
            }

            base.Initialize();
        }

        static Vector3 RandomNormalizedVector3()
        {
            return Vector3.Normalize(
                new Vector3(
                    (float)random.NextDouble() * 2 - 1, 
                    (float)random.NextDouble() * 2 - 1, 
                    (float)random.NextDouble() * 2 - 1)
                );
        }

        static Vector4 RandomVector4()
        {
            return 
                new Vector4(
                    (float)random.NextDouble(),
                    (float)random.NextDouble(),
                    (float)random.NextDouble(),
                    (float)random.NextDouble()
                );
        }

        protected override void LoadGraphicsContent(bool loadAllContent)
        {
            if (loadAllContent)
            {
                vertexDeclaration = new VertexDeclaration(
                    GraphicsDevice, 
                    FireVertex.VertexElements
                    );
                effect = content.Load<Effect>("Content/FireballShader");
                effect.Parameters["FireTexture"].SetValue(
                    content.Load<Texture2D>("Content/fire")
                    );

                effect.Parameters["ViewportHeight"].SetValue(GraphicsDevice.Viewport.Height);
                effect.Parameters["Duration"].SetValue(duration);
            }
        }

        protected override void UnloadGraphicsContent(bool unloadAllContent)
        {
            if (unloadAllContent){ content.Unload(); }
        }
        #endregion

        #region Update and Draw

        /// <summary>
        /// Allows the game component to update itself.
        /// </summary>
        /// <param name="gameTime">Provides a snapshot of timing values.</param>
        public override void Update(GameTime gameTime)
        {
            float now = (float)gameTime.TotalGameTime.TotalSeconds;
            for (int i = 0; i < vertices.Length; i++)
            {
                if (now - vertices[i].StartTime > 2)
                {
                    vertices[i].StartTime = now;
                    vertices[i].Position = new Vector3();
                    vertices[i].Velocity = RandomNormalizedVector3()*particleSpeed;
                }
            }


            base.Update(gameTime);
        }

        public override void Draw(GameTime gameTime)
        {

            GraphicsDevice.VertexDeclaration = vertexDeclaration;
            GraphicsDevice.RenderState.PointSpriteEnable = true;

            GraphicsDevice.RenderState.DepthBufferWriteEnable = false;
            GraphicsDevice.RenderState.AlphaBlendEnable = true;
            GraphicsDevice.RenderState.DestinationBlend = Blend.One;
            GraphicsDevice.RenderState.SourceBlend = Blend.SourceAlpha;

            effect.Parameters["View"].SetValue(view);
            effect.Parameters["Projection"].SetValue(projection);
            effect.Parameters["CurrentTime"].SetValue((float)gameTime.TotalGameTime.TotalSeconds);

            foreach (Fireball fireball in fireballs)
            { drawFireball(fireball); }

            GraphicsDevice.RenderState.AlphaBlendEnable = false;
            GraphicsDevice.RenderState.DepthBufferWriteEnable = true;

            base.Draw(gameTime);
        }

        private void drawFireball(Fireball fireball)
        {
            effect.Parameters["World"].SetValue(
                Matrix.CreateScale(fireball.Radius) * Matrix.CreateTranslation(fireball.Position) 
                );
            effect.Parameters["Scale"].SetValue(fireball.Radius);
            effect.Parameters["Acceleration"].SetValue(-fireball.Velocity/(duration * duration)*2);
            effect.Begin();

            foreach (EffectPass pass in effect.CurrentTechnique.Passes)
            {
                pass.Begin();

                GraphicsDevice.DrawUserPrimitives<FireVertex>(
                    PrimitiveType.PointList,
                    vertices,
                    0,
                    vertices.Length);

                pass.End();
            }

            effect.End();
        }

        #endregion

        public void SetCamera(Matrix view, Matrix projection)
        {
            this.view = view;
            this.projection = projection;
        }


        public void AddFireball(Vector3 position, float radius, Vector3 velocity)
        {
            fireballs.Add(new Fireball(position, radius, velocity));
        }

        public void ClearFireballs()
        {
            fireballs.Clear();
        }

        struct Fireball
        {
            public Vector3 Position;
            public float Radius;
            public Vector3 Velocity;

            public Fireball(Vector3 position, float radius, Vector3 velocity)
            {
                this.Position = position;
                this.Radius = radius;
                this.Velocity = velocity;
            }
        }
    }
}


