
// Shader global data

float4x4 viewProjection;
float4x4 world;
float3 lightDirection;
float3 lightColor;
float3 ambientColor;
float shadowMapSize;
float4x4 shadowMapMatrix;

texture shadowTexture;

sampler2D shadowMapSampler = sampler_state
{
	Texture = <shadowTexture>;
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU = Border;
	AddressV = Border;
	BorderColor = 0xFFFFFFFF;
}; 

texture objectTexture;

sampler2D objectTextureSampler = sampler_state
{
    Texture = <objectTexture>;
    MinFilter = Anisotropic;
    MagFilter = Anisotropic;
    MipFilter = Linear;
	AddressU = Wrap;
	AddressV = Wrap;
};




// --- Render Shadow Map Technique ---


void VS_RenderShadowMap(float4 pos : POSITION, out float4 outPos : POSITION, out float3 outPixelPos : TEXCOORD0)
{
	float4 posWorld = mul(pos, world);
	outPos = mul(posWorld, viewProjection); 
	
	outPixelPos = outPos.xyz;
}


float4 PS_RenderShadowMap(float3 pixelPos : TEXCOORD0): COLOR
{
	// write z coordinate to texture
	return pixelPos.z;
}


technique RenderShadowMap
{
	pass p0
	{
		CullMode = CW;   
		VertexShader = compile vs_2_0 VS_RenderShadowMap();
		PixelShader = compile ps_2_0 PS_RenderShadowMap();
	}
}



// --- Render Shadowed Geometry Technique ---


void VS_Shadowed(in float4 pos : POSITION, in float3 normal : NORMAL,
	float2 texCoord : TEXCOORD0, 
	out float4 outPos : POSITION,
	out float2 outTexCoord : TEXCOORD0,
	out float4 outShadowTexCoord : TEXCOORD1, 
	out float3 outDiffuse : COLOR0)
{
	float4 posWorld;
  
	posWorld = mul(pos, world);           
	outPos = mul(posWorld, viewProjection); 
  
	outDiffuse = lightColor * saturate(dot(-lightDirection, normal));

	outShadowTexCoord = mul(posWorld, shadowMapMatrix);
  
	outTexCoord = texCoord;
}


float4 PS_Shadowed(float2 texCoord : TEXCOORD0, float4 shadowTexCoord : TEXCOORD1, 
	float4 diffuse : COLOR0) : COLOR
{
	float texelSize = 1.0f / shadowMapSize;

	shadowTexCoord.xy /= shadowTexCoord.w;
  
	float4 shadow;

	shadow[0] = (shadowTexCoord.z < tex2D(shadowMapSampler, shadowTexCoord).r);
	shadow[1] = (shadowTexCoord.z < tex2D(shadowMapSampler, shadowTexCoord + float2(texelSize, 0)).r);
	shadow[2] = (shadowTexCoord.z < tex2D(shadowMapSampler, shadowTexCoord + float2(0, texelSize)).r);
	shadow[3] = (shadowTexCoord.z < tex2D(shadowMapSampler, shadowTexCoord + float2(texelSize, texelSize)).r);
  
	float2 lerpFactor = frac(shadowMapSize * shadowTexCoord);

	float lightingFactor = lerp(lerp(shadow[0], shadow[1], lerpFactor.x),
                                 lerp(shadow[2], shadow[3], lerpFactor.x),
                                 lerpFactor.y);

	diffuse *= lightingFactor;
  
	float4 outColor = 1;
	
	outColor.rgb = tex2D(objectTextureSampler, texCoord) * saturate(ambientColor + diffuse).rgb;
		
	return outColor;
}


technique Shadowed
{
	pass p0
	{
		CullMode = CCW;
		VertexShader = compile vs_2_0 VS_Shadowed();
		PixelShader = compile ps_2_0 PS_Shadowed();
	}
}

