﻿using System.Windows;
using System;

namespace SilverlightGrayScottSimulation
{
    public enum ExecutionMode
    {
        Resume,
        Pause
    }

    public class GrayScottViewModel : ViewModel
    {
        const int width = 100;
        const int height = 100;
        public GrayScottSimulator Simulator { get; private set; }
        public GrayScottViewer Viewer { get; private set; }

        public PhysicalParameterViewModel[] PhysicalParameters{get;private set;}
        public ParameterSetViewModel[] ParameterSets { get; private set; }

        private ParameterSetViewModel selectedParameters;
        public ParameterSetViewModel SelectedParameters
        {
            get { return selectedParameters; }
            set {
                this.selectedParameters = value;
                if (value == null) return;

                PhysicalParameters[0].Value = selectedParameters.Parameters.RecycleRate;
                PhysicalParameters[1].Value = selectedParameters.Parameters.ConversionRate;
                PhysicalParameters[2].Value = selectedParameters.Parameters.ActivatorDiffusionRate;
                PhysicalParameters[3].Value = selectedParameters.Parameters.InhibitorDiffusionRate;
            }
        }

        private ExecutionMode availableExecutionMode = ExecutionMode.Resume;
        public ExecutionMode AvailableExecutionMode
        {
            get
            {
                return availableExecutionMode;
            }
            set
            {
                availableExecutionMode = value;
                if (value == ExecutionMode.Pause)
                {
                    Viewer.Resume();
                }
                else
                {
                    Viewer.Pause();
                }
                NotifyPropertyChanged("AvailableExecutionMode");
            }
        }

        public MouseModeViewModel[] MouseModes { get; private set; }

        private MouseModeViewModel selectedMouseMode;
        public MouseModeViewModel SelectedMouseMode
        {
            get
            {
                return selectedMouseMode;
            }
            set
            {
                this.selectedMouseMode = value;
                NotifyPropertyChanged("SelectedMouseMode");
            }
        }

        public GrayScottViewModel()
        {
            Simulator = new GrayScottSimulator(width, height);
            Viewer = new GrayScottViewer(Simulator);

            Viewer.RunAsync();

            PhysicalParameters = new[]
            {
                new PhysicalParameterViewModel(
                    "活性因子へのリサイクル率",
                    ()=> Simulator.PhysicalParameters.RecycleRate,
                    (p)=> Simulator.PhysicalParameters.RecycleRate = p
                    ),
                new PhysicalParameterViewModel(
                    "抑制因子の分解率",
                    ()=> Simulator.PhysicalParameters.ConversionRate,
                    (p)=> Simulator.PhysicalParameters.ConversionRate = p
                    ),
                new PhysicalParameterViewModel(
                    "活性因子の拡散率",
                    ()=> Simulator.PhysicalParameters.ActivatorDiffusionRate,
                    (p)=> Simulator.PhysicalParameters.ActivatorDiffusionRate = p
                    ),
                new PhysicalParameterViewModel(
                    "抑制因子の拡散率",
                    ()=> Simulator.PhysicalParameters.InhibitorDiffusionRate,
                    (p)=> Simulator.PhysicalParameters.InhibitorDiffusionRate = p
                    )
            };

            ParameterSets = new[]
            {
                new ParameterSetViewModel("スパイラルウェーブ"){ Parameters = new PhysicalParameters(0.02, 0.05, 0.1, 0.08)},
                new ParameterSetViewModel("カオス"){ Parameters = new PhysicalParameters(0.02, 0.056, 0.18, 0.09)},
                new ParameterSetViewModel("スパイラフウェーブ"){ Parameters = new PhysicalParameters(0.02, 0.05, 0.01, 0.16)},
                new ParameterSetViewModel("自己制限集団"){ Parameters = new PhysicalParameters(0.025, 0.056, 0.18, 0.13)},
                new ParameterSetViewModel("増殖スポット"){ Parameters = new PhysicalParameters(0.035, 0.065, 0.14, 0.06)},
                new ParameterSetViewModel("ゼブラフィッシュ"){ Parameters = new PhysicalParameters(0.035, 0.06, 0.16, 0.08)},
                new ParameterSetViewModel("ワーム"){ Parameters = new PhysicalParameters(0.05, 0.065, 0.16, 0.08)},
                new ParameterSetViewModel("サンゴ"){ Parameters = new PhysicalParameters(0.062, 0.062, 0.19, 0.09)},
            };

            SelectedParameters = ParameterSets[5];

            MouseModes = new[]
            {
                new MouseModeViewModel("白塗り", p=> Viewer.MakeWhiteSpot(p, 15), ()=>Viewer.ClearSpot()),
                new MouseModeViewModel("青塗り", p=> Viewer.MakeBlueSpot(p, 15), ()=>Viewer.ClearSpot()),
                new MouseModeViewModel("壁作り", p=> Viewer.MakeWall(p), ()=>{}),
                new MouseModeViewModel("壁壊し", p=> Viewer.ClearWall(p), ()=>{})
            };

            selectedMouseMode = MouseModes[0];
        }

        public void ToggleExecutionMode()
        {
            if (AvailableExecutionMode == ExecutionMode.Resume)
            {
                AvailableExecutionMode = ExecutionMode.Pause;
            }
            else
            {
                AvailableExecutionMode = ExecutionMode.Resume;
            }
        }

        public void MouseDown(System.Windows.Input.MouseEventArgs e, System.Windows.Controls.Image image)
        {
            var position = e.GetPosition(image);
            var absPosition = new Point(
                position.X * Simulator.Width / image.ActualWidth,
                position.Y * Simulator.Height / image.ActualHeight
                );
            selectedMouseMode.MouseDown(absPosition);
        }

        public void MouseUp()
        {
            selectedMouseMode.MouseUp();
        }
    }

}
