﻿using System.Linq;
using System.Collections.Generic;

namespace SilverlightGrayScottSimulation
{

    public class Cell
    {
        public double Activator;
        public double Inhibitor;

        public double DiffuseActivator;
        public double DiffuseInhibitor;

        public List<Cell> NeighbourCells = new List<Cell>();
        public PhysicalParameters PhysicalParameters { get; set; }

        public bool IsWall;

        public void UpdateReaction()
        {
            if (IsWall) return;

            //u + 2v -> 3v
            //v -> p
            double recycleRate = PhysicalParameters.RecycleRate;//f
            double conversionRate = PhysicalParameters.ConversionRate;//k


            double reaction = Activator * Inhibitor * Inhibitor;

            double velocityActivator = -reaction + recycleRate - recycleRate * Activator;
            double velocityInhibitor = reaction - conversionRate * Inhibitor - recycleRate * Inhibitor;


            Activator += velocityActivator + DiffuseActivator;
            Inhibitor += velocityInhibitor + DiffuseInhibitor;
        }

        public void UpdateDiffuse()
        {
            NeighborInfo activatorDiffusionInfo = totalActivator();

            DiffuseActivator =
                PhysicalParameters.ActivatorDiffusionRate *
                (
                activatorDiffusionInfo.TotalChemical
                 - activatorDiffusionInfo.NonWallCellCount * Activator);

            NeighborInfo inhibitorDiffusionInfo = totalInhibitor();
            DiffuseInhibitor =
                PhysicalParameters.InhibitorDiffusionRate *
                (
                inhibitorDiffusionInfo.TotalChemical
                 - inhibitorDiffusionInfo.NonWallCellCount * Inhibitor);
        }

        private NeighborInfo totalActivator()
        {
            //NeighbourCells.Sum(n=>n.Activator)
            NeighborInfo result = new NeighborInfo { };

            for (int i = 0; i < NeighbourCells.Count; i++)
            {
                if (NeighbourCells[i].IsWall) continue;
                result.TotalChemical += NeighbourCells[i].Activator;
                result.NonWallCellCount++;
            }

            return result;
        }

        private NeighborInfo totalInhibitor()
        {
            // NeighbourCells.Sum(n=>n.Inhibitor)
            NeighborInfo result = new NeighborInfo { };

            for (int i = 0; i < NeighbourCells.Count; i++)
            {
                if (NeighbourCells[i].IsWall) continue;
                result.TotalChemical += NeighbourCells[i].Inhibitor;
                result.NonWallCellCount++;
            }

            return result;
        }
    }

    struct NeighborInfo
    {
        public int NonWallCellCount;
        public double TotalChemical;
    }
}
