﻿
namespace SilverlightGrayScottSimulation
{
    public class GrayScottSimulator
    {
        public int Width { get; private set; }
        public int Height { get; private set; }
        public Cell[,] Grid { get; private set; }
        public PhysicalParameters PhysicalParameters { get; private set; }

        public GrayScottSimulator(int width, int height)
        {
            this.Width = width;
            this.Height = height;
            Grid = new Cell[width, height];
            PhysicalParameters = new PhysicalParameters
            {
                RecycleRate = 0.02,
                ConversionRate = 0.055,
                ActivatorDiffusionRate = 0.16,
                InhibitorDiffusionRate = 0.08
            };
            initGrid();
        }

        void initGrid()
        {

            for (int x = 0; x < Width; x++)
            {
                for (int y = 0; y < Height; y++)
                {
                    Grid[x, y] = new Cell();
                    Grid[x, y].PhysicalParameters = PhysicalParameters;
                    Grid[x, y].Activator = 1;
                }
            }

            connectCells();
            initCellsAtRandom();
        }

        private void initCellsAtRandom()
        {
            System.Random random = new System.Random(0);

            for (int i = 0; i < 10; i++)
            {
                MakeSpot(
                    (int)(random.NextDouble() * Width),
                    (int)(random.NextDouble() * Height),
                    (int)(random.NextDouble() * Width / 15),
                    Lerp(0.3, 0.6, random.NextDouble()),
                    Lerp(0.2, 0.5, random.NextDouble())
                    );
            }
        }

        private void connectCells()
        {
            for (int x = 0; x < Width; x++)
            {
                for (int y = 0; y < Height; y++)
                {
                    int mostBottomY = Height - 1;
                    int mostRightX = Width - 1;

                    int topYIndex = (y != 0) ? y - 1 : mostBottomY;
                    Grid[x, y].NeighbourCells.Add(Grid[x, topYIndex]);

                    int leftXIndex = (x != 0) ? x - 1 : mostRightX;
                    Grid[x, y].NeighbourCells.Add(Grid[leftXIndex, y]);

                    int rightXIndex = (x != mostRightX) ? x + 1 : 0;
                    Grid[x, y].NeighbourCells.Add(Grid[rightXIndex, y]);

                    int bottomYIndex = (y != mostBottomY) ? y + 1 : 0;
                    Grid[x, y].NeighbourCells.Add(Grid[x, bottomYIndex]);
                }
            }
        }

        double Lerp(double a, double b, double amount)
        {
            return a + (b - a) * amount;
        }

        public void MakeSpot(int centerX, int centerY, int radius, double paintU, double paintV)
        {
            for (int y = 0; y < Height; y++)
            {
                for (int x = 0; x < Width; x++)
                {
                    int distanceFromCenter = (centerX - x) * (centerX - x) + (centerY - y) * (centerY - y);

                    if (distanceFromCenter <= radius * radius)
                    {
                        Grid[x, y].Activator = paintU;
                        Grid[x, y].Inhibitor = paintV;
                    }
                }

            }
        }


        public void Update()
        {
            //diffuse
            Parallel.For(
                0, Width,
                delegate(int x)
                {
                    for (int y = 0; y < Height; y++)
                    {
                        Cell cell = Grid[x, y];
                        cell.UpdateDiffuse();
                    }
                });

            Parallel.For(
                0, Width,
                delegate(int x)
                {
                    for (int y = 0; y < Height; y++)
                    {
                        Cell cell = Grid[x, y];
                        cell.UpdateReaction();
                    }
                });
        }

        internal void MakeWall(int x, int y)
        {
            Grid[x, y].IsWall = true;
        }

        internal void ClearWall(int x, int y)
        {
            Grid[x, y].IsWall = false;
        }
    }
}
