﻿using System;
using System.ComponentModel;
using System.Windows.Media.Imaging;

namespace SilverlightGrayScottSimulation
{
    public class GrayScottViewer
    {
        public WriteableBitmap Bitmap { get; private set; }
        byte[] bytes;

        private GrayScottSimulator simulator;
        private Spot spot;

        private bool isRunning;

        public GrayScottViewer(GrayScottSimulator simulator)
        {
            this.simulator = simulator;
            Bitmap = new WriteableBitmap(simulator.Width, simulator.Height);
            bytes = new byte[Bitmap.PixelWidth * Bitmap.PixelHeight * 4];
        }



        public void UpdatePixels()
        {
            for (int x = 0; x < simulator.Width; x++)
            {
                for (int y = 0; y < simulator.Height; y++)
                {
                    int firstIndex = (x + y * simulator.Width) * 4;

                    Cell cell = simulator.Grid[x, y];

                    if (cell.IsWall)
                    {
                        bytes[firstIndex] = byte.MaxValue;
                        bytes[firstIndex + 1] = 0;
                        bytes[firstIndex + 2] = byte.MaxValue;
                        bytes[firstIndex + 3] = byte.MaxValue;
                        continue;
                    }

                    double uPower = cell.Activator;

                    bytes[firstIndex] = (byte)((1 - cell.Inhibitor) * byte.MaxValue);
                    bytes[firstIndex + 1] = (byte)(uPower * byte.MaxValue);
                    bytes[firstIndex + 2] = (byte)(uPower * byte.MaxValue);
                    bytes[firstIndex + 3] = byte.MaxValue;
                }
            }

            for (int i = 0; i < Bitmap.Pixels.Length; i++)
            {
                int firstIndex = i * 4;
                Bitmap.Pixels[i] =
                    (bytes[firstIndex] << 0)
                     + (bytes[firstIndex + 1] << 8)
                     + (bytes[firstIndex + 2] << 16)
                     + (bytes[firstIndex + 3] << 24);
            }

            Bitmap.Invalidate();
        }

        public void RunAsync()
        {
            BackgroundWorker worker = new BackgroundWorker();
            worker.DoWork += delegate
            {
                Action updateUI = UpdatePixels;
                while (true)
                {
                    update();

                    Bitmap.Dispatcher.BeginInvoke(updateUI);
                    System.Threading.Thread.Sleep(16);
                }
            };

            worker.RunWorkerAsync();

        }

        private void update()
        {
            if (!isRunning) return;

            for (int i = 0; i < 10; i++)
            {
                if (spot != null)
                {
                    simulator.MakeSpot(
                        (int)spot.CenterPosition.X,
                        (int)spot.CenterPosition.Y,
                        spot.Radius,
                        spot.Activator,
                        spot.Inhibitor
                        );
                }
                simulator.Update();
            }
        }

        public void MakeWhiteSpot(System.Windows.Point point, int radius)
        {
            spot = new Spot { CenterPosition = point, Radius = radius, Activator = 1 };
        }

        public void MakeBlueSpot(System.Windows.Point point, int radius)
        {
            spot = new Spot { CenterPosition = point, Radius = radius, Activator = 0.5, Inhibitor = 0.5 };
        }

        public void MakeWall(System.Windows.Point point)
        {
            simulator.MakeWall((int)point.X, (int)point.Y);
        }

        public void ClearWall(System.Windows.Point point)
        {
            simulator.ClearWall((int)point.X, (int)point.Y);
        }


        public void ClearSpot() { spot = null; }

        public void Pause()
        {
            isRunning = false;
        }

        public void Resume()
        {
            isRunning = true;
        }
    }

    class Spot
    {
        public System.Windows.Point CenterPosition;
        public int Radius;
        public double Activator;
        public double Inhibitor;
    }
}
