﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using GeneticAlgorithms;

namespace SilverlightEvolutionOfRayTrackingVehicle
{
    class World : Individual
    {
        public Vehicle Vehicle { get;private set; }
        public PointLight Light = new PointLight { Quantity = 500 };
        public Genome Genome { get; set; }
        double fitness;

        public void Initialize()
        {
            Vehicle = new Vehicle();
            Vehicle.Position = new Vector2D(200, 200);

            List<LightSensor> sensors = new List<LightSensor>();
            sensors.Add(createSensor(45));
            sensors.Add(createSensor(-45));
            sensors.Add(createSensor(30));
            sensors.Add(createSensor(-30));
            sensors.Add(createSensor(15));
            sensors.Add(createSensor(-15));

            sensors.Add(createSensor(135));
            sensors.Add(createSensor(-135));

            var leftNeuron = new Neuron { };

            for (int i = 0; i < Genome.LeftWeights.Length; i++)
            {
                leftNeuron.From.Add(
                    new Connection
                    {
                        Source = sensors[i],
                        Weight = Genome.LeftWeights[i]
                    }
                    );
            }

            var rightNeuron = new Neuron { };

            for (int i = 0; i < Genome.RightWeights.Length; i++)
            {
                rightNeuron.From.Add(
                    new Connection
                    {
                        Source = sensors[i],
                        Weight = Genome.RightWeights[i]
                    }
                    );
            }

            Vehicle.LeftWheelInput = leftNeuron;
            Vehicle.RightWheelInput = rightNeuron;

            foreach (var sensor in sensors)
            {
                Vehicle.AddSensor(sensor);
            }
        }

        private LightSensor createSensor(double angleDegree)
        {
            return new LightSensor
            {
                Size = 1,
                Position = Vehicle.GetSurfacePosition(angleDegree),
                Direction = Vehicle.GetRotatedDirection(angleDegree),
                Light = Light
            };
        }

        public void Update()
        {
            Vehicle.Update();

            fitness += getLightQuantity();
        }

        private double getLightQuantity()
        {
            var lightDirection = Light.Position - Vehicle.Position;

            return Math.Min(
                Light.Quantity * Vehicle.Radius / ((Vehicle.Position - Light.Position).Length() * 2 * Math.PI),
                Light.Quantity
                );
        }

        public double GetFitness()
        {
            return fitness;
        }

        private Vector2D RandomRelativePosition(Random random, double distance)
        {
            Vector2D lightDirection = new Vector2D(
                random.NextDouble() * 2 - 1,
                random.NextDouble() * 2 - 1
                );
            return Vector2D.Normalize(lightDirection) * distance;
        }

        public void SetRandomLightPosition(Random random, double distance)
        {
            Light.Position = Vehicle.Position + RandomRelativePosition(random, distance);
        }
    }

    class WorldGAOperations : GeneticAlgorithmOperations<World>
    {
        public override World CreateAtRandom(Random random)
        {
            var result = new World { Genome = new Genome(random) };
            return result;
        }

        public override World Copy(World selected)
        {
            World result = new World { Genome = selected.Genome.Copy()};
            return result;
        }

        public override void Crossover(World a, World b, Random random)
        {
            Genome.Crossover(a.Genome, b.Genome, random);
        }

        public override void Mutate(World individual, Random random, double rate)
        {
            individual.Genome.Mutate(random, rate);
        }
    }

}