﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows;
using System.Windows.Media;

namespace SilverlightEvolutionOfRayTrackingVehicle
{

    class Vehicle
    {
        public Vector2D Position;
        public Vector2D LookDirection = new Vector2D(0, 1);

        public SignalNode LeftWheelInput;
        public SignalNode RightWheelInput;

        public readonly double Radius = 15;

        public double Rotation;

        private List<SensorHolder> sensors = new List<SensorHolder>();


        public void AddSensor(LightSensor sensor)
        {
            sensors.Add(new SensorHolder {
                Sensor = sensor,
                RelativePosition = sensor.Position - Position,
                RelativeDirection = sensor.Direction.Rotate(-Rotation)
            });
        }

        public void Update()
        {
            double LeftWheelSpeed = between(LeftWheelInput.GetValue(), -1, 1);
            double RightWheelSpeed = between(RightWheelInput.GetValue(), -1, 1);

            const double dTime = 1;
            double dLeftWheel = LeftWheelSpeed * dTime;
            double dRightWheel = RightWheelSpeed * dTime;

            Rotation += MathHelper.ToDegree(Math.Atan((dLeftWheel - dRightWheel) / (2 * Radius)));
            LookDirection = new Vector2D(0, 1).Rotate(Rotation);

            Position += LookDirection * (dLeftWheel + dRightWheel) / 2d;

            updateSensors();
        }

        private double between(double value, double min, double max)
        {
            return Math.Max(min, Math.Min(max, value));
        }

        private void updateSensors()
        {
            foreach (var sensor in sensors)
            {
                sensor.Sensor.Direction = sensor.RelativeDirection.Rotate(Rotation);
                sensor.Sensor.Position = Position + sensor.RelativePosition.Rotate(Rotation);
            }
        }

        public Vector2D GetRotatedDirection(double angleDegree)
        {
            return LookDirection.Rotate(angleDegree);
        }

        public Vector2D GetSurfacePosition(double angleDegree)
        {
            return Position + GetRotatedDirection(angleDegree) * Radius;
        }
    }

    public class SensorHolder
    {
        public Vector2D RelativeDirection;
        public Vector2D RelativePosition;
        public LightSensor Sensor;
    }
}
