﻿using System.Collections.Generic;
using System.Linq;
using System.Windows;

namespace Ecosystem
{
    public struct Collision
    {
        public Vector Sink;
        public Body Obstacle;

        public Collision(Vector sink)
        {
            this.Sink = sink;
            this.Obstacle = null;
        }
    }

    public class Body
    {
        public double Radius;
        public Vector Position;
        public Vector Velocity;
        public Vector Force;
        public List<Collision> Collisions = new List<Collision>();
        public object Tag;

        public double Top
        {
            get
            {
                return Position.Y - Radius;
            }
        }
        public double Bottom
        {
            get
            {
                return Position.Y + Radius;
            }
        }
        public double Left
        {
            get
            {
                return Position.X - Radius;
            }
        }
        public double Right
        {
            get
            {
                return Position.X + Radius;
            }
        }

        public Body(double radius)
        {
            Radius = radius;
        }

        public void Update()
        {
            double dt = 1 / 60d;
            Velocity += Force * dt;
            Position += Velocity * dt;
        }
    }

    public class PhysicalSimulator
    {
        HashSet<Body> bodies = new HashSet<Body>();
        public Rect Bounds = new Rect(0, 0, 300, 300);
        public double SpringConstant = 1000;
        public double DampingCoefficient = 2;

        public void Add(Body body)
        {
            bodies.Add(body);
        }

        public void Remove(Body body)
        {
            bodies.Remove(body);
        }

        public void Update()
        {
            var bodies = this.bodies.ToArray();

            foreach (var body in bodies)
            {
                body.Collisions.Clear();
            }

            Parallel.For(
                0, bodies.Length, 
                delegate(int i)
                {
                    var body = bodies[i];

                    if (Bounds.Right < body.Right)
                    {
                        body.Collisions.Add(new Collision (new Vector(body.Right - Bounds.Right, 0)));
                    }
                    if (body.Left < Bounds.Left)
                    {
                        body.Collisions.Add(new Collision(new Vector(body.Left - Bounds.Left, 0)));
                    }
                    if (Bounds.Bottom < body.Bottom)
                    {
                        body.Collisions.Add(new Collision(new Vector(0, body.Bottom - Bounds.Bottom)));
                    }
                    if (body.Top < Bounds.Top)
                    {
                        body.Collisions.Add(new Collision(new Vector(0, body.Top - Bounds.Top)));
                    }

                    for (int j = 0; j < bodies.Length;j++ )
                    {
                        var other = bodies[j];

                        if (body == other) { continue; }

                        var displacement = (body.Position - other.Position);

                        if (displacement.Length < (body.Radius + other.Radius))
                        {
                            var direction = displacement;

                            if (displacement.Length == 0)
                            {
                                var velocity = (body.Velocity + other.Velocity);

                                if (velocity.Length == 0) { velocity = new Vector(1, 0); }

                                direction = new Vector(velocity.Y, -velocity.X);
                                direction *= i < j ? -1 : 1;
                            }

                            direction.Normalize();

                            var collision = new Collision
                            {
                                Sink = -direction * (body.Radius + other.Radius - displacement.Length),
                                Obstacle = other
                            };
                            body.Collisions.Add(collision);
                        }
                    }
                });

            foreach (var body in bodies)
            {
                foreach (var collision in body.Collisions)
                {
                    body.Force += -collision.Sink * SpringConstant 
                        - Vector.Normalize(collision.Sink) * (body.Velocity * Vector.Normalize(collision.Sink)) * DampingCoefficient;
                }
            }

            foreach (var body in bodies)
            {
                body.Update();
                body.Force = new Vector();
            }
        }
    }

}
