/****************************************************************************
  PhotoFrame
   2010(C) Mr.Honey
****************************************************************************/

#include <stdlib.h>
#include "board.h"
#include "sdc.h"

// Module selector (SPI2)
#define SPIMOD	PMD1bits.SPI2MD
#define SPICON1	SPI2CON1bits
#define SPICON2	SPI2CON2bits
#define SPISTAT	SPI2STATbits
#define SPIBUF	SPI2BUF
#define DMACON	DMA2CON
#define DMAREQ	DMA2REQ
#define DMASTA	DMA2STA
#define DMAPAD	DMA2PAD
#define DMACNT	DMA2CNT
#define DMAIF	IFS1bits.DMA2IF
#define DMAIE	IEC1bits.DMA2IE
#define CHEN	DMA2CONbits.CHEN
#define SPIIRQ	33
#define SDC_CS	SDC2_CS
#define ReadSDC	ReadSDC2

// SDC Commands
#define GO_IDLE_STATE		0x40
#define SEND_OP_COND		0x41
#define SEND_IF_COND		0x48
#define SEND_CSD			0x49
#define SET_BLOCKLEN		0x50
#define READ_SINGLE_BLOCK	0x51
#define WRITE_SINGLE_BLOCK	0x58
#define APP_OP_COND			0x69
#define APP_CMD				0x77
#define READ_OCR			0x7A

#define FLOATING_BUS		0xFF
#define DATA_START_TOKEN	0xFE
#define DATA_ACCEPTED		0b00000101

#define MODE_NORMAL			9
#define MODE_HC				0

#define Clocking8()			SendSPI(0xFF)
#define NOCRC				0xFF

//=================================================================

static void InitSPI(void);
static U8 CheckSDC(void);
static U8 BusyCheck(void);
static U8 SendCMD(U8 cmd, U32 param, U8 crc, U8 f1, U8 f2);
static U8 RecvSPI(void);
static U8 SendSPI(U8 data);

//=================================================================

static int s_nMode;
static int s_nBusy;

//=================================================================

// 
extern U8 OpenSDC2(void)
{
	int nCount;
	U8 nR1;

	if (SPISTAT.SPIEN)
		return RET_NG;

	s_nMode = -1;
	s_nBusy = FALSE;

	// W[ON
	SPIMOD = 0;

	// C^[tF[X
	InitSPI();
	DelayMs(1);

	nCount = 10;
	do Clocking8();
	while (--nCount > 0);

	// J[hZbg
	SDC_CS = 0;
	DelayMs(1);
	nR1 = SendCMD(GO_IDLE_STATE, 0, 0x95, TRUE, TRUE);
	if (nR1 != 0x01)
	{
		SPISTAT.SPIEN = 0;
		return RET_NG;
	}

	// J[h
	nR1 = SendCMD(SEND_IF_COND, 0x1AA, 0x87, FALSE, FALSE);
	if (nR1 <= 0x01)
	{
		// SDC Ver2 or SDHC
		U32 nR7 =
			((U32)RecvSPI() << 24) |
			((U32)RecvSPI() << 16) |
			((U32)RecvSPI() <<  8) |
			((U32)RecvSPI() <<  0);

		Clocking8();
		SDC_CS = 1;

		s_nMode = MODE_HC;

		if (nR7 == 0x1AA)
		{
			nCount = 500;
			do {
				DelayMs(1);
				SendCMD(APP_CMD, 0, NOCRC, TRUE, FALSE);
				nR1 = SendCMD(APP_OP_COND, 0x40000000, NOCRC, TRUE, TRUE);
			}
			while(nR1 != 0x00 && --nCount > 0);
		}
		else {
			nCount = 0;
		}
	}
	else
	{
		// MMC or SDC Ver1
		Clocking8();
		SDC_CS = 1;

		s_nMode = MODE_NORMAL;

		nCount = 500;
		do {
			DelayMs(1);
			nR1 = SendCMD(SEND_OP_COND, 0, 0xF9, TRUE, TRUE);
		}
		while(nR1 != 0x00 && --nCount > 0);
	}

	// J[hݒmF
	if (nCount == 0 || CheckSDC() != RET_OK)
	{
		SPISTAT.SPIEN = 0;
		return RET_NG;
	}

	return RET_OK;
}

// ZN^[h
extern U8 ReadSDC2(U32 addr, U8* buf)
{
	int nCount;
	U8 nR1;

	// rW[`FbN
	if (BusyCheck() != RET_OK)
		return RET_NG;

	// R}hs
	if (SendCMD(READ_SINGLE_BLOCK, addr << s_nMode, NOCRC, TRUE, FALSE) != 0x00)
	{
		SDC_CS = 1;
		if (SendCMD(READ_SINGLE_BLOCK, addr << s_nMode, NOCRC, TRUE, FALSE) != 0x00)
		{
			SDC_CS = 1;
			return RET_NG;
		}
	}

	// g[NmF
	nCount = 4000;
	do nR1 = RecvSPI();
	while (nR1 == FLOATING_BUS && --nCount > 0);

	if (nR1 == DATA_START_TOKEN)
	{
		// DMA ZbgAbv
		DMACS1 = 0;
		DMACON = 0x4001; // SIZE:1 DIR:0 HALF:0 AMODE:00 MODE01
		DMAREQ = SPIIRQ;
		DMASTA = (U16)buf - (U16)&_DMA_BASE;
		DMAPAD = (volatile U16)&SPIBUF;
		DMACNT = SDCBLK - 1;
		DMAIE = 1;
		CHEN = 1;

		// SCLKNbNo
		nCount = SDCBLK;
		do {
			SPIBUF = 0xFF;
			while(SPISTAT.SPITBF);
		}
		while (--nCount > 0);
		
		// DMAI҂
		while (DMAIE);

		// CRC ǂ
		RecvSPI();
		RecvSPI();
	}

	Clocking8();
	SDC_CS = 1;

	return (nR1 == DATA_START_TOKEN)? RET_OK : RET_NG;
}

// ZN^Cg
extern U8 WriteSDC2(U32 addr, const U8* buf)
{
	U8 nDO;
	int i;

	// rW[`FbN
	if (BusyCheck() != RET_OK)
		return RET_NG;

	// R}hs
	if (SendCMD(WRITE_SINGLE_BLOCK, addr << s_nMode, NOCRC, TRUE, FALSE) != 0x00)
	{
		SDC_CS = 1;
		return RET_NG;
	}

	// g[No
	SendSPI(DATA_START_TOKEN);

	// f[^o
	i = 0;
	do {
		unsigned nTemp;
		SPIBUF = buf[i];
		while(!SPISTAT.SPIRBF);
		nTemp = SPIBUF;
	}
	while (++i < SDCBLK);

	// CRC 󑗏o
	SendSPI(NOCRC);
	SendSPI(NOCRC);

	// X|XmF
	nDO = RecvSPI() & 0x0F;
	if (nDO == DATA_ACCEPTED)
		s_nBusy = TRUE;
        
	Clocking8();
	SDC_CS = 1;

	return (nDO == DATA_ACCEPTED)? RET_OK : RET_NG;
}

// 
extern void CloseSDC2(void)
{
	// rW[`FbNiȂꍇ͋Ij
	BusyCheck();

	// SDI/SDO/SCK͖(fW^̓|[g)Ƃ
	// dfɔ
	SPISTAT.SPIEN = 0;

	// W[OFF
	SPIMOD = 1;
}

//=================================================================

// SPI
static void InitSPI(void)
{
	U16 nTemp;

	// dsPIC33FJ256GP710
	// SPI Mode0 - 353.69kHz(Fcy:39.61375MHz)
	SPICON1.DISSCK = 0;
	SPICON1.DISSDO = 0;
	SPICON1.MODE16 = 0;
	SPICON1.SMP = 0;
	SPICON1.CKE = 1;
	SPICON1.SSEN = 0;
	SPICON1.CKP = 0;
	SPICON1.MSTEN = 1;
	SPICON1.SPRE = 0b001;
	SPICON1.PPRE = 0b01;
	SPICON2.FRMEN = 0;
	SPISTAT.SPISIDL = 1;
	SPISTAT.SPIROV = 0;
	SPISTAT.SPIEN = 1;

	nTemp = SPIBUF;
}

// J[hݒmF
static U8 CheckSDC(void)
{
	// ]xto
	SPISTAT.SPIEN = 0;
	SPICON1.PPRE = 0b11;
	//SPICON1.SPRE = 0b100; // Fcy/4=9.9MHz
	SPICON1.SPRE = 0b101; // Fcy/3=13.2MHz
	SPISTAT.SPIEN = 1;

	// AhbVOmF
	if (s_nMode == MODE_HC)
	{
		s_nMode = MODE_NORMAL;
		if (SendCMD(READ_OCR, 0, NOCRC, FALSE, FALSE) == 0x00)
		{
			U32 nR7 =
				((U32)RecvSPI() << 24) |
				((U32)RecvSPI() << 16) |
				((U32)RecvSPI() <<  8) |
				((U32)RecvSPI() <<  0);

			if ((nR7 & 0xC0000000) == 0xC0000000)
				s_nMode = MODE_HC;
		}
		Clocking8();
		SDC_CS = 1;
	}

	// ubNTCYݒ
	if (s_nMode == MODE_NORMAL)
	{
		if (SendCMD(SET_BLOCKLEN, SDCBLK, NOCRC, TRUE, TRUE) != 0x00)
			return RET_NG;
	}

	return RET_OK;
}

// rW[`FbN
static U8 BusyCheck(void)
{
	if (s_nBusy)
	{
		U32 nCount;
		U8 nDO;

		SDC_CS = 0;

		nCount = 0xFFFFF;
		do nDO = RecvSPI();
		while (nDO == 0x00 && --nCount > 0);

		Clocking8();
		SDC_CS = 1;

		if (nCount == 0)
			return RET_NG;

		s_nBusy = FALSE;
	}

	return RET_OK;
}

// R}hM
static U8 SendCMD(U8 cmd, U32 param, U8 crc, U8 f1, U8 f2)
{
	int nCount;
	U8 nR1;
	    
	SDC_CS = 0;

	// R}hubNM
	SendSPI(cmd);
	SendSPI((U8)(param >> 24));
	SendSPI((U8)(param >> 16));
	SendSPI((U8)(param >> 8));
	SendSPI((U8)(param >> 0));
	SendSPI(crc);

	// R1X|XM
	nCount = 8;
	do nR1 = RecvSPI();
	while ((nR1 & 0x80) != 0 && --nCount > 0);
    
	// 㑱
	if (f1) Clocking8();
	if (f2) SDC_CS = 1;

	return (nCount > 0)? nR1 : FLOATING_BUS;
}

// SPIM
static U8 RecvSPI(void)
{
	SPIBUF = 0xFF;
	while(!SPISTAT.SPIRBF);
	return (U8)SPIBUF;
}

// SPIM
static U8 SendSPI(U8 data)
{
	SPIBUF = data;
	while(!SPISTAT.SPIRBF);
	return (U8)SPIBUF;
}

// DMA2
void _ISRNOPSV _DMA2Interrupt(void)
{
	DMAIE = 0;
	DMAIF = 0;
}
