/****************************************************************************

	HoneyRadio
		Copyright(C) 2012 Mr.Honey

****************************************************************************/

#include "Main.h"
#include "Display.h"
#include "SDC.h"
#include "Volume.h"
#include "Decoder.h"
#include "Network.h"
#include "FMRadio.h"
#include "AMRadio.h"
#include "NetRadio.h"
#include "Player.h"
#include "User.h"

//=================================================================

// Configuration
// DEVCFG0
#pragma config DEBUG = ON			// Debugger is enabled 
#pragma config ICESEL = ICS_PGx2	// ICE EMUC2/EMUD2 pins shared with PGC2/PGD2
#pragma config PWP = OFF			// Program Flash Write Protect: Disabled
#pragma config BWP = OFF			// Boot Flash Write Protect: Disabled
#pragma config CP = OFF				// Code Protect: Protection: Disabled

// DEVCFG1
#pragma config FNOSC = PRIPLL		// Primary oscillator (XT, HS, EC) w/ PLL
#pragma config FSOSCEN = OFF		// Secondary Oscillator Enable: Disabled
#pragma config IESO = OFF			// Internal/External Switch Over: Disabled
#pragma config POSCMOD = HS			// Primary Oscillator Configuration: HS osc mode (8MHz)
#pragma config OSCIOFNC = OFF		// CLKO Output Signal Active on the OSCO Pin: Disabled
#pragma config FPBDIV = DIV_1		// Peripheral Clock Divisor: Divide by 1
#pragma config FCKSM = CSECMD		// Clock Switching Enabled, Clock Monitoring Disabled
#pragma config WDTPS = PS2048		// Watchdog Timer Postscaler: 1:1024
#pragma config FWDTEN = OFF			// WDT Disabled (SWDTEN Bit Controls)

// DEVCFG2
#pragma config FPLLIDIV = DIV_2		// Divide by 2 (4MHz)
#pragma config FPLLMUL = MUL_20		// Multiply by 20 (80MHz)
#pragma config UPLLEN = OFF			// USB PLL Enable: Disabled and Bypassed
#pragma config FPLLODIV = DIV_2		// Divide by 1 (40MHz)

// DEVCFG3
#pragma config FUSBIDIO = OFF		// USB USID Selection: Controlled by Port Function 
#pragma config FVBUSONIO = OFF		// USB VBUS ON Selection: Controlled by Port Function

//=================================================================

#define ST_PWOFF		0
#define ST_OFCHG		1
#define ST_ONCHG		2
#define ST_POWON		3

#define APP_OFF			(-1)
#define APP_FMR			0
#define APP_AMR			1
#define APP_NTR			2
#define APP_MP3			3

#define BAT_FULL		2
#define BAT_LEVEL2		1
#define BAT_NORMAL		0
#define BAT_LEVEL1		(-1)
#define BAT_LOW			(-2)
#define BAT_EMPTY		(-3)

#define BLV_LEVEL2		784	// 10.3V(1.47V * 7)
#define BLV_NORMAL		746	//  9.8V(1.40V * 7)
#define BLV_LEVEL1		670	//  8.8V(1.26V * 7)
#define BLV_LOW			640	//  8.4V(1.20V * 7)
#define BLV_EMPTY		624	//  8.2V(1.17V * 7)

#define DATAPATH		"\\"
#define DATAFILE		"SETTINGS.DAT"
#define SETTINGFILE		"\\SETTINGS.INI"

#define CHARGE_TIME		(12 * 3600 * 1000) // ő12
#define SWHOLD_TIME		700 // ms
#define AUTORP_TIME		250 // ms

//#define DISABLE_BAT // for Debug
//#define DISABLE_PSW // for Debug

#ifndef DISABLE_PSW
#define D_POWERSW_IO	POWERSW_IO
#else
#define D_POWERSW_IO	0
#endif

//=================================================================

static void InitSystem(void);
static void Run(void);
static void Shutdown(void);

static void OnPower(void);
static void OnAdpOn(void);
static void OnCommon(void);
static void OnLowBat(void);

static void WaitPSW(void);
static BOOL CheckBat(void);
static void StartChg(void);
static void StopChg(void);
static BOOL IsEndChg(void);
static void OffTimer(void);

static void MainTask(void);
static void StartApp(void);
static void StopApp(void);
static void Indicator(void);
static void OnClock(DWORD dwOld, DWORD dwNew);

//=================================================================
datas_t g_Data = { 0 };
DWORD g_dwSysClock = DEFAULT_CLK;
BYTE g_Buffer[BUFFER_SIZE];

static DWORD s_dwSave = 0;

static int s_nState = ST_PWOFF;
static BOOL s_fAdpOn = FALSE;
static int s_nBattery = BAT_NORMAL;
static DWORD s_dwBtTick = 0;
static BOOL s_fEndChg = FALSE;
static int s_nAppMode = APP_OFF;
static int s_nOffTimer = 0;
static DWORD s_dwOffTick = 0;

static int s_nInput;
static int s_nInput1;
static int s_nInput2;
static int s_nInput3;
static DWORD s_dwInTick = 0;

static int s_nRemain1 = 0;
static int s_nRemain2 = 0;
static int s_nT3Unit;

static int s_nIndA;
static int s_nIndB;
static int s_nIndC;
static int s_nIndD;

//=================================================================

// CGg
int main(void)
{
	InitSystem();

	TickInit();
	InitDisplay();
	InitVolume();
	InitSDC();
	InitNet();
	InitFMR();
	InitAMR();
	InitDecoder();
	InitNTR();
	InitMP3();

	Run();

	for(;;) Shutdown();
}

// n[h
static void InitSystem(void)
{
	// JTAG
	DDPCONbits.JTAGEN = 0;

	// }`xN^荞
	INTEnableSystemMultiVectoredInt();
	
	// ptH[}XœK
	SYSTEMConfigPerformance(GetSystemClock());
	mOSCSetPBDIV(OSC_PB_DIV_1);

	// |[g
	LATB = LATC = LATD = LATE = LATF = LATG = 0;
	AD1PCFG = 0xFFFFFFFF;

	// Other ports
	POW_MAIN_IO = 1; // Power latch up
	POW_MAIN_TRIS = OUTPUT;
	POW_PAMP_IO = 0;
	POW_PAMP_TRIS = OUTPUT;
	MAINLED_IO = 1;
	MAINLED_TRIS = OUTPUT;
	SPMUTE_IO = 0;
	SPMUTE_TRIS = OUTPUT;
	CHARGE_IO = 0;
	CHARGE_TRIS = OUTPUT;
	POWERSW_TRIS = INPUT;
	ADAPTER_TRIS = INPUT;
	BATV_TRIS = INPUT;
	BATV_PCFG = ANALOG;
	CNPUESET = SW_CNMASK;
	SW_TRISSET = SW_IOMASK;

	CNENSET = POWERSW_CNMASK;
	CNCONbits.ON = 1;
	IPC6bits.CNIP = 2;
	IEC1bits.CNIE = 1;

	// ADC(TvO,ϊ,ADRC)
	AD1CON1bits.SIDL = 0;
	AD1CON1bits.FORM = 0b100;
	AD1CON1bits.SSRC = 0b111;
	AD1CON1bits.CLRASAM = 0;
	AD1CON1bits.ASAM = 1;
	AD1CON2 = 0;
	AD1CON3bits.ADRC = 1;
	AD1CON3bits.SAMC = 4;

	// ChargeTimer(T3)
	T3CON = 0;
	T3CONbits.TCKPS = 0b111; // 1:256
	PR3 = 62500; // 40MHz 0.4b
	IPC3bits.T3IP = 1;
	IEC0bits.T3IE = 1;
	s_nT3Unit = 400;

	// DelayTimer(T4/T5)
	T4CON = T5CON = 0;
	T4CONbits.T32 = 1;
	IEC0bits.T5IE = 1;

	// 荞݋
	INTEnableInterrupts();

	// ҂(RDA5800CZbgԊ܂)
	DelayMS(50);
}

// s[v
static void Run(void)
{
#ifndef __DEBUG
	EnableWDT();
#endif
	// dCxg(POWSW|ADPON)
	CheckBat();
	if (POWERSW_IO || !ADAPTER_IO) {
		OnPower();
	}
	else {
		OnAdpOn();
	}

	// C[v
	while (s_nState != ST_PWOFF)
	{
		ClearWDT();

		if (D_POWERSW_IO)
		{
			// dSW
			OnPower();
		}
		else if (ADAPTER_IO != s_fAdpOn)
		{
			// A_v^}
			if (s_fAdpOn) {
				OnCommon();
			}
			else {
				OnAdpOn();
			}
			s_fAdpOn = !s_fAdpOn;
		}
		else if (IsEndChg())
		{
			// [dI
			OnCommon();
		}
		else if (CheckBat())
		{
			// obed
			if (s_nBattery < BAT_NORMAL) {
				OnLowBat();
			}
			else if (s_nBattery > BAT_LEVEL2) {
				OnCommon();
			}
		}
		// C^XN
		OffTimer();
		MainTask();
	}
}

// Vbg_E
static void Shutdown(void)
{
	ASSERT(s_nState == ST_PWOFF && s_nAppMode == APP_OFF);

	MAINLED_IO = 0;

	// dr؂\
	if (s_nBattery <= BAT_EMPTY)
	{
		ClearDisplay();
		DisplayText(0, 0, "  Low Battery.  ");
		DisplayCtrl(DISPLAY_ON);
		DelayMS(3000);
		DisplayCtrl(DISPLAY_RESET);
	}
	
	// dOFF
	INTDisableInterrupts();
	POW_MAIN_IO = 0;
	PowerSaveSleep();
}

//=================================================================

// POWSWCxg
static void OnPower(void)
{
	WaitPSW();
	switch (s_nState)
	{
	case ST_PWOFF:
		if (s_nBattery > BAT_EMPTY) {
			s_nState = ST_POWON;
		}
		else if (ADAPTER_IO) {
			s_nState = ST_POWON;
		}
		break;

	case ST_OFCHG:
		s_nState = ST_ONCHG;
		break;

	case ST_ONCHG:
		s_nState = ST_OFCHG;
		break;

	case ST_POWON:
		s_nState = ST_PWOFF;
		break;

	default:
		ASSERT(FALSE);
		break;
	}
}

// ADPONCxg
static void OnAdpOn(void)
{
	switch (s_nState)
	{
	case ST_PWOFF:
		if (s_nBattery < BAT_NORMAL)
		{
			StartChg();
			s_nState = ST_OFCHG;
		}
		break;

	case ST_POWON:
		if (s_nBattery < BAT_NORMAL)
		{
			StartChg();
			s_nState = ST_ONCHG;
		}
		break;

	default:
		ASSERT(FALSE);
	}
}

// ADOFF/FULBAT/ENDCHGCxg
static void OnCommon(void)
{
	switch (s_nState)
	{
	case ST_OFCHG:
		StopChg();
		s_nState = ST_PWOFF;
		break;

	case ST_ONCHG:
		StopChg();
		s_nState = ST_POWON;
		break;

	case ST_POWON:
		break;

	default:
		ASSERT(FALSE);
	}
}

// LOWBATCxg
static void OnLowBat()
{
	switch (s_nState)
	{
	case ST_OFCHG:
	case ST_ONCHG:
		break;

	case ST_POWON:
		if (s_fAdpOn)
		{
			if (!s_fEndChg)
			{
				StartChg();
				s_nState = ST_ONCHG;
			}
			break;
		}
		if (s_nBattery <= BAT_EMPTY) {
			s_nState = ST_PWOFF;
		}
		break;

	default:
		ASSERT(FALSE);
	}
}

// dSW𗣂܂ő҂
static void WaitPSW(void)
{
	int i;
	for (i = 0; i < 50; i++)
	{
		if (D_POWERSW_IO) {
			i = 0;
		}
	}
}

// obe[`FbN
static BOOL CheckBat(void)
{
#ifndef DISABLE_BAT

	static DWORD nVals[4];
	static int nCount = 0;

	if (TickGet() >= s_dwBtTick || s_dwBtTick == 0)
	{
		// 4񕽋ϒl߂
		int nVal = 0, nBat, i;

		AD1CHSbits.CH0SA = BATV_AN;

		AD1CON1bits.ON = 1;
		DelayUS(250);
		AD1CON1bits.DONE = 0;
		while(!AD1CON1bits.DONE); // ͓ǂݎ̂Ă
		do {
			AD1CON1bits.DONE = 0;
			while(!AD1CON1bits.DONE);
			nVals[nCount % 4] = ADC1BUF0;
		}
		while (++nCount < 4);
		AD1CON1bits.ON = 0;

		for (i = 0; i < 4; i++) {
			nVal += nVals[i];
		}
		nVal /= 4;

		// 4iK\֕ϊ
		if (nVal < BLV_EMPTY) {
			nBat = BAT_EMPTY;
		}
		else if (nVal < BLV_LOW) {
			nBat = BAT_LOW;
		}
		else if (nVal < BLV_LEVEL1) {
			nBat = BAT_LEVEL1;
		}
		else if (nVal < BLV_NORMAL) {
			nBat = BAT_NORMAL;
		}
		else if (nVal < BLV_LEVEL2) {
			nBat = BAT_LEVEL2;
		}
		else {
			nBat = BAT_FULL;
		}

		// Tick(3b)
		s_dwBtTick = TickGet() + TICKS_OF(3);

		// ω`FbN
		if (nBat != s_nBattery || nBat == BAT_EMPTY)
		{
			s_nBattery = nBat;
			return TRUE;
		}
	}
#endif
	return FALSE;
}

// [dJn
static void StartChg(void)
{
	ASSERT(IEC0bits.T3IE);
	ASSERT(ADAPTER_IO && !CHARGE_IO && s_nRemain1 == 0);

	CHARGE_IO = 1;
	s_nRemain1 = CHARGE_TIME;
	T3CONbits.TON = 1;
}

// [d~
static void StopChg(void)
{
	ASSERT(CHARGE_IO);
	CHARGE_IO = 0;
	s_nRemain1 = 0;
}

// [dI`FbN
static BOOL IsEndChg(void)
{
	ASSERT(!CHARGE_IO || (T3CONbits.TON && IEC0bits.T3IE));
	if (CHARGE_IO && s_nRemain1 <= 0)
	{
		s_fEndChg = TRUE;
		return TRUE;
	}
	return FALSE;
}

// C^XN
static void MainTask(void)
{
	// e@\̌ďo֐
	static void (*Starts[])(void) = {
		StartFMR, StartAMR, StartNTR, StartMP3, StartUser
	};
	static void (*Tasks[])(void) = {
		TaskFMR, TaskAMR, TaskNTR, TaskMP3, TaskUser
	};
	static void (*Stops[])(void) = {
		StopFMR, StopAMR, StopNTR, StopMP3, StopUser
	};
	int nNewMode;

	// [hؑ֔f
	if (s_nState == ST_POWON || s_nState == ST_ONCHG)
	{
		if (s_nAppMode == APP_OFF)
		{
			StartApp();
			nNewMode = g_Data.nLastMode;
		}
		else
		{
			// [h֑ؑ
			nNewMode = s_nAppMode;
			if (GetPushingSW(SW_MODE))
			{
				if (IsDisplay()) {
					nNewMode = (s_nAppMode + 1) % 5;
				}
				else {
					DisplayCtrl(DISPLAY_AUTO);
				}
			}
		}
	}
	else {
		nNewMode = APP_OFF;
	}

	if (s_nAppMode != nNewMode)
	{
		// [hؑ֎s
		if (s_nAppMode != APP_OFF)
		{
			// ݂̃[hI
			Stops[s_nAppMode]();
			ASSERT(g_dwSysClock == DEFAULT_CLK);
		}
		if (nNewMode != APP_OFF)
		{
			// V[hJn
			ASSERT(g_dwSysClock == DEFAULT_CLK);
			Starts[nNewMode]();
			ResetInput();
		}
		else {
			StopApp();
		}

		s_nAppMode = nNewMode;
	}
	else
	{
		if (s_nAppMode != APP_OFF)
		{
			// {[
			switch (GetPushingSW(SW_VUP | SW_VDOWN))
			{
			case SW_VUP:
				SetVolume(g_Data.nLastVolume + 1);
				g_Data.nLastVolume = (BYTE)GetVolume();
				break;
			case SW_VDOWN:
				SetVolume(g_Data.nLastVolume - 1);
				g_Data.nLastVolume = (BYTE)GetVolume();
				break;
			}
			// e@\^XNR[
			ASSERT(s_nState == ST_POWON || s_nState == ST_ONCHG);
			Tasks[s_nAppMode]();

			// Ɏs^XN
			Indicator();
			TaskDisplay();
		}
		else
		{
			ASSERT(s_nState == ST_OFCHG || s_nState == ST_PWOFF);

			// OFF[dX[v
			if (s_nState == ST_OFCHG)
			{
				MAINLED_IO = ~MAINLED_IO; // [d_
				TaskSleep();
			}
		}
	}
}

// AvN
static void StartApp(void)
{
	ASSERT(s_nAppMode == APP_OFF);
	ASSERT(IsMute() && !SPMUTE_IO && !POW_PAMP_IO);
	ASSERT(IEC1bits.CNIE && (CNEN & SW_CNMASK) == 0);
	ASSERT(s_nOffTimer == 0 && s_nRemain2 == 0 && s_dwOffTick == 0);

	MAINLED_IO = 1;

	// LCD\
	DisplayText(0, 0, "HoneyRadio (^^)/");
	DisplayCtrl(DISPLAY_AUTO);

	s_nIndA = 0;
	s_nIndB = 0;
	s_nIndC = 0;
	s_nIndD = 0x80;

	// AvdON
	SPMUTE_IO = 1;
	DelayMS(50);
	POW_PAMP_IO = 1;
	DelayMS(800);
	SPMUTE_IO = 0;

	// ݒ蕜
	LoadData();
	SetVolume(g_Data.nLastVolume);

	// SW͋
	CNENSET = SW_CNMASK;

	DisplayText(11, 0, "     ");
}

// AvI
static void StopApp(void)
{
	ASSERT(s_nAppMode != APP_OFF);
	ASSERT(IsMute() && !SPMUTE_IO && POW_PAMP_IO);
	ASSERT(IEC1bits.CNIE && (CNEN & SW_CNMASK) == SW_CNMASK);

	// LCD
	DisplayCtrl(DISPLAY_RESET);
	ClearDisplay();

	// AvdOFF
	POW_PAMP_IO = 0;

	// ݒۑ
	g_Data.nLastMode = (BYTE)s_nAppMode;
	SaveData();

	// SW͋֎~(PowerSW)
	CNENCLR = SW_CNMASK;
	ResetInput();

	// It^C}[
	SetOffTimer(FALSE);
}

// It^C}[
static void OffTimer(void)
{
	ASSERT(0 <= s_nOffTimer && s_nOffTimer <= 4);
	ASSERT(s_nRemain2 <= (60 * 60 * 1000));

	if (s_nState == ST_ONCHG || s_nState == ST_POWON)
	{
		if (s_nAppMode != APP_OFF)
		{
			if (s_nOffTimer != 0 && s_nRemain2 <= 0)
			{
				if (TickGet() >= s_dwOffTick || s_dwOffTick == 0)
				{
					SetVolume(GetVolume() - 1);
					if (GetVolume() == 0) {
						s_nState = (s_nState == ST_ONCHG)? ST_OFCHG : ST_PWOFF;
					}
					else {
						s_dwOffTick = TickGet() + TICKS_OF(3);
					}
				}
			}
		}
	}
}

// CWP[^[
static void Indicator(void)
{
	// It^C}[
	int nMin = (s_nRemain2 + 60 * 1000 - 1) / (60 * 1000);
	if (s_nOffTimer != s_nIndA || nMin != s_nIndB)
	{
		if (s_nOffTimer != 0)
		{
			char buf[8];
			sprintf(buf, "#%02u", nMin);
			DisplayText(11, 0, buf);
		}
		else {
			DisplayText(11, 0, "   ");
		}
		s_nIndA = s_nOffTimer;
		s_nIndB = nMin;
	}

	// [d\
	if (CHARGE_IO != s_nIndC)
	{
		DisplayText(14, 0, (CHARGE_IO)? "\x7E" : " ");
		s_nIndC = CHARGE_IO;
	}

	// obe[\
	ASSERT((BAT_FULL - BAT_EMPTY) == 5);
	ASSERT(BAT_EMPTY <= s_nBattery && s_nBattery <= BAT_FULL);
	static const char* BatChars[] = { "E", "L", "1", "2", "F", "F" };
	if (s_nBattery != s_nIndD)
	{
		DisplayText(15, 0, BatChars[s_nBattery - BAT_EMPTY]);
		s_nIndD = s_nBattery;
	}

	// _͂Ȃ
}

//=================================================================

// NbNݒ
extern void SetClock(DWORD dwClock)
{
	if (g_dwSysClock != dwClock)
	{
		int nSource = OSC_POSC_PLL;
		int nPost = OSC_PLL_POST_1;

		switch (dwClock)
		{
		case CLK_80MHz: nPost = OSC_PLL_POST_1; break;
		case CLK_40MHz: nPost = OSC_PLL_POST_2; break;
		case CLK_20MHz: nPost = OSC_PLL_POST_4; break;
		case CLK_10MHz: nPost = OSC_PLL_POST_8; break;
		case CLK_8MHz: nSource = OSC_POSC; break;
		case CLK_5MHz: nPost = OSC_PLL_POST_16; break;
		case CLK_2MHz: nPost = OSC_PLL_POST_32; break;
		case CLK_1MHz: nPost = OSC_PLL_POST_64; break;
		case CLK_32KHz: nSource = OSC_LPRC; break;
		default:
			ASSERT(FALSE);
		}

		if (dwClock > g_dwSysClock) {
			// ɂȂ鎞͐ݒO
			SYSTEMConfigPerformance(dwClock);
		}
		else {
			// Ɏs
			OnClock(g_dwSysClock, dwClock);
		}

		OSCConfig(nSource, OSC_PLL_MULT_20, nPost, OSC_FRC_POST_1);

		if (dwClock < g_dwSysClock) {
			// ᑬɂȂ͐ݒ
			SYSTEMConfigPerformance(dwClock);
		}
		else {
			// Ɏs
			OnClock(g_dwSysClock, dwClock);
		}

		g_dwSysClock = dwClock;

		// SDC{[[gĐݒ
		OnClockSDC();
	}
}

// NbNݒ(ꎞύXp)
extern void PushClock(DWORD dwClock)
{
	s_dwSave = g_dwSysClock;
	SetClock(dwClock);
}

// NbN(ꎞύXp)
extern void PopClock(void)
{
	ASSERT(s_dwSave != 0);
	SetClock(s_dwSave);
	s_dwSave = 0;
}

// NbNύXɔ
static void OnClock(DWORD dwOld, DWORD dwNew)
{
	// TMR3Đݒ
	IEC0CLR = _IEC0_T3IE_MASK;
	int nSave = T3CONbits.TON;
	T3CONbits.TON = 0;

	int nRemain = (s_nT3Unit * TMR3) / PR3;
	if (s_nRemain1 > 0) {
		s_nRemain1 -= nRemain;
	}
	if (s_nRemain2 > 0) {
		s_nRemain2 -= nRemain;
	}
	TMR3 = 0;
	switch (dwNew)
	{
	case CLK_80MHz: PR3 = 62500, T3CONbits.TCKPS = 0b111; s_nT3Unit = 200; break;
	case CLK_40MHz: PR3 = 62500; T3CONbits.TCKPS = 0b111; s_nT3Unit = 400; break;
	case CLK_20MHz: PR3 = 62500; T3CONbits.TCKPS = 0b111; s_nT3Unit = 800; break;
	case CLK_10MHz: PR3 = 31250; T3CONbits.TCKPS = 0b111; s_nT3Unit = 800; break;
	case  CLK_8MHz: PR3 = 31250; T3CONbits.TCKPS = 0b111; s_nT3Unit = 1000; break;
	case  CLK_5MHz: PR3 = 62500; T3CONbits.TCKPS = 0b110; s_nT3Unit = 800; break;
	case  CLK_2MHz: PR3 = 62500; T3CONbits.TCKPS = 0b101; s_nT3Unit = 800; break;
	case  CLK_1MHz: PR3 = 62500; T3CONbits.TCKPS = 0b100; s_nT3Unit = 800; break;
	case CLK_32KHz: PR3 = 32768; T3CONbits.TCKPS = 0b000; s_nT3Unit = 1000; break;
	default:
		ASSERT(FALSE);
	}
#ifdef __DEBUG
	if (T3CONbits.TCKPS != 0b111)
		ASSERT(s_nT3Unit == ((long long)PR3 * 1000 * (1 << T3CONbits.TCKPS)) / dwNew);
	else
		ASSERT(s_nT3Unit == ((long long)PR3 * 1000 * 256) / dwNew);
#endif
	IFS0CLR = _IFS0_T3IF_MASK;
	IEC0SET = _IEC0_T3IE_MASK;
	T3CONbits.TON = nSave;

	// NbNύXʒm
	OnClockDisp(dwOld, dwNew);
	OnClockAMR(dwOld, dwNew);

	// L^ĂTicklC
	if (s_dwInTick != 0)
	{
		DWORD dwTick = TickGet() + 1;
		s_dwInTick = dwTick - ((dwTick - s_dwInTick) * (long long)dwNew) / dwOld;
	}
	if (s_dwBtTick != 0)
	{
		DWORD dwTick = TickGet() + 1;
		s_dwBtTick = (s_dwBtTick > dwTick)?
			(dwTick + ((s_dwBtTick - dwTick) * (long long)dwNew) / dwOld) : dwTick;
	}
	if (s_dwOffTick != 0)
	{
		DWORD dwTick = TickGet() + 1;
		s_dwOffTick = (s_dwOffTick > dwTick)?
			(dwTick + ((s_dwOffTick - dwTick) * (long long)dwNew) / dwOld) : dwTick;
	}
}

// It^C}[ݒ
extern void SetOffTimer(BOOL fSet)
{
	if (fSet)
	{
		ASSERT(s_nAppMode != APP_OFF);
	
		if ((s_nOffTimer * 15) - (s_nRemain2 / (60 * 1000)) < 2)
		{
			// ݂̐ݒ15₷
			s_nOffTimer = (s_nOffTimer + 1) % 5;
			s_nRemain2 = s_nOffTimer * (15 * 60 * 1000);
			s_dwOffTick = 0;
		}
		else
		{
			// ݂̐ݒɖ߂
			s_nRemain2 = s_nOffTimer * (15 * 60 * 1000);
			s_dwOffTick = 0;
		}
		T3CONbits.TON = 1;
		DisplayCtrl(DISPLAY_AUTO);
	}
	else
	{
		// \͍XVȂ(Ip)
		s_nOffTimer = 0;
		s_nRemain2 = 0;
		s_dwOffTick = 0;
	}
}
// SW͎擾()
extern int GetPushingSW(int nSWs)
{
	ASSERT(IEC1bits.CNIE && (CNEN & SW_CNMASK) == SW_CNMASK);

	IEC1CLR = _IEC1_CNIE_MASK;
	Nop();
	int nRet = SW_NONE;
	if (s_nInput1 & nSWs)
	{
		nRet = s_nInput1;
		s_nInput1 = SW_NONE;
	}
	else if (s_nInput & nSWs)
	{
		// I[gs[g
		DWORD dwTemp = TickGet();
		ASSERT(s_dwInTick != 0);
		if ((dwTemp - s_dwInTick) >= TICKS_OFMS(SWHOLD_TIME + AUTORP_TIME))
		{
			nRet = s_nInput;
			s_dwInTick = dwTemp - TICKS_OFMS(SWHOLD_TIME);
		}
	}
	IEC1SET = _IEC1_CNIE_MASK;

	return nRet;
}

// SW͎擾()
extern int GetHoldingSW(int nSWs)
{
	ASSERT(IEC1bits.CNIE && (CNEN & SW_CNMASK) == SW_CNMASK);

	IEC1CLR = _IEC1_CNIE_MASK;
	Nop();
	int nRet = SW_NONE;
	if (s_nInput & nSWs)
	{
		ASSERT(s_dwInTick != 0);
		if ((TickGet() - s_dwInTick) >= TICKS_OFMS(SWHOLD_TIME))
		{
			nRet = s_nInput;
			s_nInput2 = SW_NONE;
		}
	}
	IEC1SET = _IEC1_CNIE_MASK;

	return nRet;
}

// SW͎擾(Z)
extern int GetPressedSW(int nSWs)
{
	ASSERT(IEC1bits.CNIE && (CNEN & SW_CNMASK) == SW_CNMASK);

	IEC1CLR = _IEC1_CNIE_MASK;
	Nop();
	int nRet = SW_NONE;
	if (s_nInput3 & nSWs)
	{
		nRet = s_nInput3;
		s_nInput3 = SW_NONE;
	}
	IEC1SET = _IEC1_CNIE_MASK;

	return nRet;
}

// SW̓NA
extern void ResetInput(void)
{
	s_nInput = SW_NONE;
	s_nInput1 = SW_NONE;
	s_nInput2 = SW_NONE;
	s_nInput3 = SW_NONE;
	s_dwInTick = 0;

	IFS1bits.CNIF = 0;
}

// X[v
extern void TaskSleep(void)
{
	ASSERT(IEC1bits.CNIE && CNEN != 0);
	ASSERT(s_nAppMode == APP_OFF && s_nRemain2 == 0 && s_dwInTick == 0);

	// 2048msX[v
	ClearWDT();
	PowerSaveSleep();

	// [dԂ猸Z
	if (s_nRemain1 > 0) {
		s_nRemain1 -= 2048 + 1; // ms
	}

	// obe`FbNpTickZbg
	s_dwBtTick = 0;
}

// ݐ
extern void INTCtrl(BOOL fEnable)
{
	if (fEnable)
	{
		IEC0SET = _IEC0_T3IE_MASK;
		IEC1SET = _IEC1_CNIE_MASK;
	}
	else
	{
		IEC0CLR = _IEC0_T3IE_MASK;
		IEC1CLR = _IEC1_CNIE_MASK;
	}
}

// fBC(ms)
extern void DelayMS(DWORD dwMs)
{
	ASSERT(IEC0bits.T5IE);

	DWORD nCount = (GetPeripheralClock() / 1000UL * dwMs);
	if (nCount == 0) {
		return;
	}
	else if (nCount > 101) {
		nCount -= 100;
	}
	PR4 = (WORD)nCount;
	PR5 = (WORD)(nCount >> 16);
	TMR4 = TMR5 = 0;

	IFS0CLR = _IFS0_T5IF_MASK;
	T4CONSET = _T4CON_ON_MASK;
	while (!IFS0bits.T5IF);
	T4CONCLR = _T4CON_ON_MASK;
}

// fBC(us)
extern void DelayUS(DWORD dwUs)
{
	ASSERT(IEC0bits.T5IE);

	DWORD nCount = (GetPeripheralClock() / 1000000UL * dwUs);
	if (nCount == 0) {
		return;
	}
	else if (nCount > 101) {
		nCount -= 100;
	}
	
	PR4 = (WORD)nCount;
	PR5 = (WORD)(nCount >> 16);
	TMR4 = TMR5 = 0;

	IFS0CLR = _IFS0_T5IF_MASK;
	T4CONSET = _T4CON_ON_MASK;
	while (!IFS0bits.T5IF);
	T4CONCLR = _T4CON_ON_MASK;
}

// ݒt@CJ
extern BOOL OpenSettings(setting_t* pObj)
{
	VERIFY(f_chdrive(DRV_SDC) == FR_OK);
	VERIFY(f_chdir("\\") == FR_OK);
	return (f_open(&pObj->file, SETTINGFILE, FA_READ) == FR_OK);
}

// ݒǍ()
extern BOOL ReadSettingStr(setting_t* pObj, const char* pKey)
{
	int i;
	for (i = (f_tell(&pObj->file) == 0)? 1 : 0; i < 2; i++)
	{
		while (f_gets(pObj->buf, sizeof(pObj->buf), &pObj->file) != NULL)
		{
			if (pObj->buf[0] != '#') // Rgs
			{
				char* p;
				const char* k = pKey;
				for (p = pObj->buf; *k != '\0' && *p != '\0'; k++, p++)
				{
					if (*k == *p)
					{
						if (p[1] == '=') // Name=Value`
						{
							int nLen = strlen(p += 2);
							if (nLen > 0 && p[nLen - 1] == '\n')
								p[nLen - 1] = '\0';

							pObj->Str = p;
							return TRUE;
						}
						continue;
					}
					break;
				}
			}
		}
		if (f_lseek(&pObj->file, 0) != FR_OK) {
			break;
		}
	}

	return FALSE;
}

// ݒǍ(10il)
extern BOOL ReadSettingNum(setting_t* pObj, const char* pKey)
{
	if (ReadSettingStr(pObj, pKey))
	{
		pObj->Num = atoi(pObj->Str);
		return TRUE;
	}
	return FALSE;
}

// ݒǍ(IP/MACAhX)
extern BOOL ReadSettingAddr(setting_t* pObj, const char* pKey, BYTE nRadix)
{
	int i;
	for (i = (f_tell(&pObj->file) == 0)? 1 : 0; i < 2; i++)
	{
		while (f_gets(pObj->buf, sizeof(pObj->buf), &pObj->file) != NULL)
		{
			if (pObj->buf[0] != '#') // Rgs
			{
				char* p;
				const char *e, *k = pKey;
				for (p = pObj->buf; *k != '\0' && *p != '\0'; k++, p++)
				{
					if (*k == *p)
					{
						if (p[1] == '=') // Name=Value`
						{
							BYTE* n = pObj->Addr;
							BYTE* s = n + sizeof(pObj->Addr);

							int nLen = strlen(p += 2);
							if (nLen > 0 && p[nLen - 1] == '\n')
								p[nLen - 1] = '\0';

							memset(n, 0, sizeof(pObj->Addr));
							for (; *p != '\0' && n < s; n++)
							{
								for(e = p; *e != '.' && *e != '\0'; e++);
								for(; p < e; p++) {
									BYTE v = ('0' <= *p && *p <= '9')? (*p - '0') :
										(('A' <= *p && *p <= 'F')? (*p - 'A' + 10) :
											(('a' <= *p && *p <= 'f')? (*p - 'a' + 10) : 0));
									*n = *n * nRadix + v;
								}
								if (*p == '.') p++;
							}
							return TRUE;
						}
						continue;
					}
					break;
				}
			}
		}
		if (f_lseek(&pObj->file, 0) != FR_OK) {
			break;
		}
	}

	return FALSE;
}

// ݒt@C
extern void CloseSettings(setting_t* pObj)
{
	ASSERT(IsOpenSDC());
	VERIFY(f_close(&pObj->file) == FR_OK);
}

// f[^[h
extern void LoadData(void)
{
	VERIFY(f_chdrive(DRV_SDC) == FR_OK);
	VERIFY(f_chdir(DATAPATH) == FR_OK);

	FIL file;
	if (f_open(&file, DATAFILE, FA_READ) == FR_OK)
	{
		datas_t buf;
		UINT rbytes;
		if (f_read (&file, &buf, sizeof(buf), &rbytes) == FR_OK)
		{
			if (rbytes == sizeof(buf))
			{
				g_Data = buf;
				VERIFY(f_close(&file) == FR_OK);
				return;
			}
		}
		VERIFY(f_close(&file) == FR_OK);
	}

	// t@Cǂ߂Ȃꍇ̓t@C쐬Ă
	if (f_open(&file, DATAFILE, FA_CREATE_ALWAYS | FA_WRITE) == FR_OK)
	{
		UINT wbytes;
		VERIFY(f_write(&file, &g_Data, sizeof(g_Data), &wbytes) == FR_OK);
		VERIFY(f_close(&file) == FR_OK);
	}
}

// f[^Z[u
extern void SaveData(void)
{
	VERIFY(f_chdrive(DRV_SDC) == FR_OK);
	VERIFY(f_chdir(DATAPATH) == FR_OK);
	
	// t@C݂鎞ۑ
	// ̂ŁA鎞FTPŃt@C폜ēdOFFΗǂ
	FIL file;
	if (f_open(&file, DATAFILE, FA_OPEN_EXISTING | FA_WRITE) == FR_OK)
	{
		UINT wbytes;
		VERIFY(f_write (&file, &g_Data, sizeof(g_Data), &wbytes) == FR_OK);
		VERIFY(f_close(&file) == FR_OK);
	}
}

// Oo
extern void OutputLog(const char* pLog)
{
	// LCD2sڂɕ\
	DisplayText(0, 1, "                ");
	DisplayText(0, 1, pLog);
	DisplayCtrl(DISPLAY_TEMPON);
}

// AT[V
#ifdef __DEBUG
extern void _Assert(BOOL cond, const char* msg, int line)
{
	if (!cond)
	{
		char buf[64];

		ClearDisplay();
		DisplayText(0, 0, "Assertion!");
		sprintf(buf, "L:%d F:%s", line, msg);
		DisplayText(0, 1, buf);
		DisplayCtrl(DISPLAY_ON);

		for(;;);
	}
}
#endif //__DEBUG

//=================================================================

// TMR3
void __ISR(_TIMER_3_VECTOR, ipl1) _T3Interrupt(void)
{
	TMR3 = 0;
	if (s_nRemain1 > 0) {
		s_nRemain1 -= s_nT3Unit;
	}
	if (s_nRemain2 > 0) {
		s_nRemain2 -= s_nT3Unit;
	}
	IFS0CLR = _IFS0_T3IF_MASK;
}

// CN
void __ISR(_CHANGE_NOTICE_VECTOR, ipl2) _CNInterrupt(void)
{
	const int SwVals[] = { SW_VUP, SW_VDOWN, SW_NEXT, SW_PREV, SW_FUNC, SW_MODE };

	// ͒l擾(`^O)
	WORD wInput = 0;
	int a, b, i, j = g_dwSysClock / 3276;
	for (a = 0, b = 0, i = 0; i < j; i++)
	{
		WORD nVal = ~SW_IO & SW_IOMASK;
		if (nVal != 0)
		{
			wInput |= nVal;
			a++;
		}
		else {
			b++;
		}
	}
	if (a > b)
	{
		// 
		if (s_nInput == SW_NONE)
		{
			// オɔ
			s_dwInTick = TickGet();
			s_nInput = SwVals[wInput - 1];
			s_nInput1 = s_nInput;
			s_nInput2 = s_nInput;
			s_nInput3 = SW_NONE;
		}
	}
	else
	{
		// J
		s_nInput3 = s_nInput2;
		s_nInput1 = SW_NONE;
		s_nInput2 = SW_NONE;
		s_nInput = SW_NONE;
		s_dwInTick = 0;
	}

	IFS1CLR = _IFS1_CNIF_MASK;
}

// C30 and C32 Exception Handlers
void _general_exception_handler(unsigned cause, unsigned status)
{
	static enum
	{
		EXCEP_IRQ = 0,		// interrupt
		EXCEP_AdEL = 4,		// address error exception (load or ifetch)
		EXCEP_AdES,			// address error exception (store)
		EXCEP_IBE,			// bus error (ifetch)
		EXCEP_DBE,			// bus error (load/store)
		EXCEP_Sys,			// syscall
		EXCEP_Bp,			// breakpoint
		EXCEP_RI,			// reserved instruction
		EXCEP_CpU,			// coprocessor unusable
		EXCEP_Overflow,		// arithmetic overflow
		EXCEP_Trap,			// trap (possible divide by zero)
		EXCEP_IS1 = 16,		// implementation specfic 1
		EXCEP_CEU,			// CorExtend Unuseable
		EXCEP_C2E			// coprocessor 2
	} _excep_code;

	static UINT _excep_addr;

	char buf[32];
	asm volatile("mfc0 %0,$13" : "=r" (_excep_code));
	asm volatile("mfc0 %0,$14" : "=r" (_excep_addr));
	_excep_code = (_excep_code & 0x0000007C) >> 2;

	ClearDisplay();
	DisplayText(0, 0, "Exception!");
	sprintf(buf, "A:%08X C:%d", _excep_addr, _excep_code);
	DisplayText(0, 1, buf);
	DisplayCtrl(DISPLAY_ON);

	for(;;);
}
