package collada;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import javafx.geometry.Point3D;
import javafx.scene.Group;
import javafx.scene.shape.Sphere;
import javafx.scene.transform.Affine;
import javafx.scene.transform.Transform;

public class DaeNode extends DaeObject
{
    private Group           root            = new Group();          //
    private DaeGeometry     geometry        = null;                 //
    
    private String          type            = null;
    private Sphere          sphere          = new Sphere();         // ジョイント位置を示す球体
    private List<DaeNode>   children        = new ArrayList<>();    // 子ノード
    private Affine          transform       = new Affine();         // このノードの変換。デフォルトは単位行列
    private Affine          parentTransform = new Affine();         // 親ノードの変換。デフォルトは単位行列
    private Affine          globalTransform = new Affine();         // 親＋このノードの変換合成。デフォルトは単位行列
    
    /**
     * 変換行列を変換として登録
     * @param matrix
     */
    public void setMatrix(double[][] matrix)
    {
        // アフィン変換（拡大・縮小、回転、平行移動）を作成
        transform  = new Affine( matrix[0][0] , matrix[0][1] , matrix[0][2] , matrix[0][3] , 
                                 matrix[1][0] , matrix[1][1] , matrix[1][2] , matrix[1][3] , 
                                 matrix[2][0] , matrix[2][1] , matrix[2][2] , matrix[2][3]  );
        
        // 自分自身の変換とマージ
        sphere.getTransforms().clear();
        sphere.getTransforms().add( parentTransform );
        sphere.getTransforms().add( transform );
        
        // 関連付けられたメッシュに変換を適用
        if( geometry != null )
        {
            root    = geometry.getRoot();
            root.getTransforms().add( parentTransform );
            root.getTransforms().add( transform );
        }
        
        // 変換を合成
        globalTransform      = new Affine();
        for( Transform transform : sphere.getTransforms() )
        { globalTransform.append( transform ); }
        
        // 子ノードに変換を再設定
        for( DaeNode child  : children )
        { child.setParentTransform( globalTransform ); }
        
    }
    
    /**
     * ジョイントの配列を取得
     * @return
     */
    public List<DaeNodeJoint> getJoints()
    {
        // ノードの中からJOINTだけを抽出
        List<DaeNodeJoint> joints  = children.stream()
                                          .filter( o -> o instanceof DaeNodeJoint )
                                          .map( o -> (DaeNodeJoint) o )
                                          .collect( Collectors.toList() );
        return joints;
    }
    
    /**
     * 親の変換＋子の変換を取得する
     * @return
     */
    public Affine getGrobalTransforms() {
        return globalTransform;
    }

    /**
     * 親の変換を登録
     * @param parentTransform
     */
    public void setParentTransform( Affine parentTransform )
    {
        // 親の変換を登録
        this.parentTransform = parentTransform;
        
        // 自分自身の変換とマージ
        sphere.getTransforms().clear();
        sphere.getTransforms().add( parentTransform );
        sphere.getTransforms().add( transform );

        // 関連付けられたメッシュに変換を適用
        if( geometry != null )
        {
            root    = geometry.getRoot();
            root.getTransforms().add( parentTransform );
            root.getTransforms().add( transform );
        }
        
        // 変換を合成
        globalTransform      = new Affine();
        for( Transform transform : sphere.getTransforms() )
        { globalTransform.append( transform ); }
        
        // 子ノードに変換を再設定
        for( DaeNode child  : children )
        { child.setParentTransform( globalTransform ); }
        
    }
    

    /**
     * ノードの中心座標を計算して返します
     * @return
     */
    public Point3D getPoint3D(){
        
        // 単位行列を作成
        Affine  e           = new Affine( 1 , 0 , 0 , 0 ,
                                          0 , 1 , 0 , 0 ,
                                          0 , 0 , 1 , 0 );
       
        // 単位行列に変換を掛け合わせる
        for( Transform transform : sphere.getTransforms() )
        { e.append( transform ); }
        
        double  x   = e.getTx();
        double  y   = e.getTy();
        double  z   = e.getTz();

        return new Point3D( x , y , z );
    }
    
    /* ここから下は単純なsetter/getter */


    public void setGeometry(DaeGeometry geometry) {
        this.root       = geometry.getRoot();
        this.geometry   = geometry;
    }

    public Group getRoot() {
        return root;
    }
    
    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }
        public Sphere getSphere() {
        return sphere;
    }
    public List<DaeNode> getChildren() {
        return children;
    }
    public Sphere getNode() {
        return sphere;
    }
}
