package collada;

import java.util.HashMap;
import java.util.Map;

import javafx.geometry.Point3D;
import javafx.scene.Camera;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.PerspectiveCamera;
import javafx.scene.paint.Color;
import javafx.scene.paint.PhongMaterial;
import javafx.scene.shape.Cylinder;
import javafx.scene.shape.DrawMode;
import javafx.scene.shape.Shape3D;
import javafx.scene.shape.Sphere;
import javafx.scene.transform.Rotate;
import javafx.scene.transform.Translate;

/**
 * Colladaファイル内のSceneオブジェクト
 * @author tomo
 *
 */
public class DaeVisualScene extends DaeObject
{
    // 内部変数
    Camera              camera          = new PerspectiveCamera(true);
    DrawMode            drawMode        = DrawMode.FILL;

    // ジョイント関係
    DaeNode             daeNodeRoot     = null;             // DaeNodeのルート
    Map<String,DaeNode> daeNodeMap      = new HashMap<>();  // DaeNodeのマップ
    Group               meshRootNode    = null;
    Group               jointsRootNode  = null;      //

    /**
     * 表示用メッシュを作成する
     * @return
     */
    public Node getMeshNode()
    {
        // メッシュ作成前の場合のみ、メッシュを作成する
        if( meshRootNode == null )
        {
            // 新しくルートノードを作成
            this.meshRootNode   = new Group();
            
            // ノードをたどり、メッシュを取得
            makeMeshNodeRecursive( this.meshRootNode , daeNodeRoot );
        }
        
        return this.meshRootNode;
    }
    
    /**
     * 表示用メッシュを再帰的に取得する
     * @param out
     * @param parentNode
     */
    private void makeMeshNodeRecursive( Group out , DaeNode parentNode )
    {
        // 子ノードが存在しない場合処理終了確認（再帰の終了条件）
        if( ( parentNode == null ) || ( parentNode.getChildren().size() == 0 ) ){ return; }
        
        // メッシュを追加する。
        for( DaeNode node : parentNode.getChildren() )
        {
            // メッシュを取得
            if( ( node.getRoot() != null ) )
            { out.getChildren().add( node.getRoot() ); }
            
            // 下層を探索
            makeMeshNodeRecursive( out , node );
            
        } 
    }
    
    /**
     * ドローモード（面描写か線描写か）を変更する
     * @param mode
     */
    public void setDrawMode( DrawMode mode )
    {
        // クラス変数を変更
        this.drawMode   = mode;
        
        // ルートノードに対して
        // DrawModeを変更する再帰処理を開始
        setDrawModeRecursive( meshRootNode , this.drawMode );
    }
    
    /**
     * ドローモードを変更する再帰処理
     * @param node
     * @param mode
     */
    private void setDrawModeRecursive( Node node , DrawMode mode )
    {
        // ノードの型を判定
        if( node instanceof Group )
        {   // 型変換
            Group   group   = (Group) node;
            
            // 子が存在しない場合は停止（再帰ツリーの末端）
            if( group.getChildren().size() == 0 ){ return; }
            
            // さらに下の階層を探索
            for( Node o : group.getChildren() ){ setDrawModeRecursive( o , mode ); }
            
        }else if( node instanceof Shape3D ){
            // 型変換
            Shape3D shape   = (Shape3D) node;
            
            // モードを設定
            shape.setDrawMode( mode );
            
        }else{
            throw new RuntimeException( "DrawModeの設定に失敗しました。" );
        }

    }
    
    /**
     * 画面に出力するジョイントイメージを作成する
     * @return
     */
    public Node getJointsNode()
    {
        // ジョイントメッシュを作成していない場合のみ
        // ジョイントメッシュを作成
        if( jointsRootNode == null )
        {
            // ルートノードを作成
            jointsRootNode  = new Group();

            // ジョイントノードを再帰的に探索し、
            // ルートノードに登録していく
            makeJointsNodeRecursive( jointsRootNode , daeNodeRoot );
        }
        
        return jointsRootNode;
    }
    
    /**
     * ジョイントを表示するかを設定する
     * @param flg
     */
    public void setJointVisible( Boolean flg )
    {
        jointsRootNode.setVisible( flg );
    }
    
    /**
     * ジョイントを再帰的に探索し、描画ノードを登録していく
     * @param parent
     */
    private void makeJointsNodeRecursive( Group out , DaeNode parentNode )
    {
        // 子ノードが存在しない場合処理終了確認（再帰の終了条件）
        if( ( parentNode == null ) || ( parentNode.getChildren().size() == 0 ) ){ return; }
        
        // ジョイントを追加する。
        for( DaeNode node : parentNode.getChildren() )
        {
            // ジョイントの場合は位置を示す球体を出力
            if( node instanceof DaeNodeJoint )
            { 
                // 球体を取得
                Sphere  sphere  = node.getSphere(); 
                sphere.setRadius( 0.05 );
                
                // 色を指定
                PhongMaterial   material    = new PhongMaterial();
                material.setDiffuseColor( Color.RED );
                sphere.setMaterial( material );
                
                // 描写ノードとして登録
                out.getChildren().add( sphere );
                
                // 親がジョイントの場合は、親子のジョイント間を線でつないでボーンを表現する
                if( parentNode instanceof DaeNodeJoint )
                {
                    // 親と子の座標を取得
                    Point3D coordParent     = parentNode.getPoint3D();
                    Point3D coordChild      = node.getPoint3D();
                    
                    // 2点間の中点と距離を計算
                    Point3D centerPoint     = coordParent.midpoint( coordChild );
                    double  distance        = coordParent.distance( coordChild );
                    
                    // 回転軸を計算
                    // y軸と親子をつなぐ線に直角な線を外積を求めることで計算
                    Point3D rotateAxis      = coordParent.subtract( coordChild ).crossProduct( 0.0 , 1.0 , 0.0 );
                    
                    // 回転角を計算
                    double  rotateAngle     = coordParent.subtract( coordChild ).angle( 0.0 , 1.0 , 0.0 );
                    
                    // 線分を作成
                    Cylinder line           = new Cylinder( 0.025 , distance );
                    line.getTransforms().add( new Translate( centerPoint.getX() , centerPoint.getY() , centerPoint.getZ() ) );
                    line.getTransforms().add( new Rotate( - rotateAngle , rotateAxis ) );
                    
                    // 色を設定
                    line.setMaterial( material );
                    
                    out.getChildren().add( line );
                }
                
            }
            
            // 下層を探索
            makeJointsNodeRecursive( out , node );
            
        } 
    }

    // 以下は単純なgetter/setter
    
    public DaeNode getDaeNodeRoot() {
        return daeNodeRoot;
    }

    public void setDaeNodeRoot(DaeNode daeNodeRoot) {
        this.daeNodeRoot = daeNodeRoot;
    }

    public Map<String, DaeNode> getDaeNodeMap() {
        return daeNodeMap;
    }

    public Camera getCamera() {
        return camera;
    }

    public void setCamera(Camera camera) {
        this.camera = camera;
    }

    public DrawMode getDrawMode() {
        return drawMode;
    }
}
