// $Id: evaluate.c,v 1.6 2010-04-09 08:24:27 eiki Exp $

#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "bonasse.h"
#include "shogi.h"

#define DBG_DIFEVAL 0

//#define INANIWA_SHIFT

#ifndef DBG_NO_ATOMIC64

void ld64b_atomic64(uint64_t* p, uint64_t* a) {
 unsigned u0, u1;
 uint64_t u;
 __m128i x = _mm_loadl_epi64((__m128i*)p);
 u0 = (unsigned int)_mm_cvtsi128_si32(x);
 x = _mm_srli_si128(x, 4);  // >> 32b(4B)
 u1 = (unsigned int)_mm_cvtsi128_si32(x);
 u = u1;
 *a = (u << 32) | u0;
}

void ld128b_atomic64(volatile trans_entry_t* p, uint64_t* w1, uint64_t* w2) {
 unsigned u0, u1, u2, u3;
 uint64_t uh, ul;
 __m128i x = _mm_load_si128((__m128i*)p);
 u0 = (unsigned int)_mm_cvtsi128_si32(x);
 x = _mm_srli_si128(x, 4);  // >> 32b(4B)
 u1 = (unsigned int)_mm_cvtsi128_si32(x);
 x = _mm_srli_si128(x, 4);  // >> 32b(4B)
 u2 = (unsigned int)_mm_cvtsi128_si32(x);
 x = _mm_srli_si128(x, 4);  // >> 32b(4B)
 u3 = (unsigned int)_mm_cvtsi128_si32(x);
 uh = u3;
 *w2 = (uh << 32) | u2;
 ul = u1;
 *w1 = (ul << 32) | u0;
}

void st64b_atomic64(uint64_t* p, uint64_t a) {
 __m128i x = _mm_set_epi32(0, 0, (int)(a >> 32), (int)(a & 0xffffffff));
 _mm_storel_epi64((__m128i*)p, x);
}

void st128b_atomic64(volatile trans_entry_t* p, uint64_t w1, uint64_t w2) {
 __m128i x = _mm_set_epi32( (int)(w2 >> 32), (int)(w2 & 0xffffffff),
                            (int)(w1 >> 32), (int)(w1 & 0xffffffff));
 _mm_store_si128((__m128i*)p, x);
}

#endif

static char kpptransstr[32][20] = {
 "f_hand_pawn   ",
 "e_hand_pawn   ",
 "f_hand_lance  ",
 "e_hand_lance  ",
 "f_hand_knight ",
 "e_hand_knight ",
 "f_hand_silver ",
 "e_hand_silver ",
 "f_hand_gold   ",
 "e_hand_gold   ",
 "f_hand_bishop ",
 "e_hand_bishop ",
 "f_hand_rook   ",
 "e_hand_rook   ",
 "f_pawn        ",
 "e_pawn        ",
 "f_lance       ",
 "e_lance       ",
 "f_knight      ",
 "e_knight      ",
 "f_silver      ",
 "e_silver      ",
 "f_gold        ",
 "e_gold        ",
 "f_bishop      ",
 "e_bishop      ",
 "f_horse       ",
 "e_horse       ",
 "f_rook        ",
 "e_rook        ",
 "f_dragon      ",
 "e_dragon      " 
};

static int kpptranslimit[33] = {
  f_hand_pawn   ,
  e_hand_pawn   ,
  f_hand_lance  ,
  e_hand_lance  ,
  f_hand_knight ,
  e_hand_knight ,
  f_hand_silver ,
  e_hand_silver ,
  f_hand_gold   ,
  e_hand_gold   ,
  f_hand_bishop ,
  e_hand_bishop ,
  f_hand_rook   ,
  e_hand_rook   ,
  f_pawn        ,
  e_pawn        ,
  f_lance       ,
  e_lance       ,
  f_knight      ,
  e_knight      ,
  f_silver      ,
  e_silver      ,
  f_gold        ,
  e_gold        ,
  f_bishop      ,
  e_bishop      ,
  f_horse       ,
  e_horse       ,
  f_rook        ,
  e_rook        ,
  f_dragon      ,
  e_dragon      ,
  fe_end
};

static int kpptranslimit2[33] = {
  f_hand_pawn   ,
  e_hand_pawn   ,
  f_hand_lance  ,
  e_hand_lance  ,
  f_hand_knight ,
  e_hand_knight ,
  f_hand_silver ,
  e_hand_silver ,
  f_hand_gold   ,
  e_hand_gold   ,
  f_hand_bishop ,
  e_hand_bishop ,
  f_hand_rook   ,
  e_hand_rook   ,
  f_pawn+9      ,
  e_pawn        ,
  f_lance+9     ,
  e_lance       ,
  f_knight+18   ,
  e_knight      ,
  f_silver      ,
  e_silver      ,
  f_gold        ,
  e_gold        ,
  f_bishop      ,
  e_bishop      ,
  f_horse       ,
  e_horse       ,
  f_rook        ,
  e_rook        ,
  f_dragon      ,
  e_dragon      ,
  fe_end
};

static int sq2x(int sq) { return (9-sq%9); }
static int sq2y(int sq) { return (1+sq/9); }
static char kpptransary2[20];

static char* kpptrans(int x) {
 int i, sq, suf = -1;
 for(i=0; i<33; i++)
   if (x < kpptranslimit2[i])
     { suf = i-1; break; }
 assert(suf != -1);
 sq = x - kpptranslimit[suf];
 if (suf<14)
   sprintf(&kpptransstr[suf][14], "%2d", sq);
 else
   sprintf(&kpptransstr[suf][14], "%d%d", sq2x(sq), sq2y(sq));
 return &kpptransstr[suf][0];
}

static char* kpptrans2(int x) {
 int i, sq, suf = -1;
 for(i=0; i<33; i++)
   if (x < kpptranslimit2[i])
     { suf = i-1; break; }
 assert(suf != -1);
 sq = x - kpptranslimit[suf];
 strcpy(kpptransary2, &kpptransstr[suf][0]);
 if (suf<14)
   sprintf(&kpptransary2[14], "%2d", sq);
 else
   sprintf(&kpptransary2[14], "%d%d", sq2x(sq), sq2y(sq));
 return kpptransary2;
}


static int ehash_probe( uint64_t current_key, unsigned int hand_b,
			int *pscore );
static void ehash_store( uint64_t key, unsigned int hand_b, int score );
static int make_list( const tree_t * restrict ptree, int * restrict pscore,
		      int list0[52], int list1[52] );

int
eval_material( const tree_t * restrict ptree )
{
  int material, itemp;

  itemp     = PopuCount( BB_BPAWN )   + (int)I2HandPawn( HAND_B );
  itemp    -= PopuCount( BB_WPAWN )   + (int)I2HandPawn( HAND_W );
  material  = itemp * p_value[15+pawn];

  itemp     = PopuCount( BB_BLANCE )  + (int)I2HandLance( HAND_B );
  itemp    -= PopuCount( BB_WLANCE )  + (int)I2HandLance( HAND_W );
  material += itemp * p_value[15+lance];

  itemp     = PopuCount( BB_BKNIGHT ) + (int)I2HandKnight( HAND_B );
  itemp    -= PopuCount( BB_WKNIGHT ) + (int)I2HandKnight( HAND_W );
  material += itemp * p_value[15+knight];

  itemp     = PopuCount( BB_BSILVER ) + (int)I2HandSilver( HAND_B );
  itemp    -= PopuCount( BB_WSILVER ) + (int)I2HandSilver( HAND_W );
  material += itemp * p_value[15+silver];

  itemp     = PopuCount( BB_BGOLD )   + (int)I2HandGold( HAND_B );
  itemp    -= PopuCount( BB_WGOLD )   + (int)I2HandGold( HAND_W );
  material += itemp * p_value[15+gold];

  itemp     = PopuCount( BB_BBISHOP ) + (int)I2HandBishop( HAND_B );
  itemp    -= PopuCount( BB_WBISHOP ) + (int)I2HandBishop( HAND_W );
  material += itemp * p_value[15+bishop];

  itemp     = PopuCount( BB_BROOK )   + (int)I2HandRook( HAND_B );
  itemp    -= PopuCount( BB_WROOK )   + (int)I2HandRook( HAND_W );
  material += itemp * p_value[15+rook];

  itemp     = PopuCount( BB_BPRO_PAWN );
  itemp    -= PopuCount( BB_WPRO_PAWN );
  material += itemp * p_value[15+pro_pawn];

  itemp     = PopuCount( BB_BPRO_LANCE );
  itemp    -= PopuCount( BB_WPRO_LANCE );
  material += itemp * p_value[15+pro_lance];

  itemp     = PopuCount( BB_BPRO_KNIGHT );
  itemp    -= PopuCount( BB_WPRO_KNIGHT );
  material += itemp * p_value[15+pro_knight];

  itemp     = PopuCount( BB_BPRO_SILVER );
  itemp    -= PopuCount( BB_WPRO_SILVER );
  material += itemp * p_value[15+pro_silver];

  itemp     = PopuCount( BB_BHORSE );
  itemp    -= PopuCount( BB_WHORSE );
  material += itemp * p_value[15+horse];

  itemp     = PopuCount( BB_BDRAGON );
  itemp    -= PopuCount( BB_WDRAGON );
  material += itemp * p_value[15+dragon];

  return material;
}

 //****

#ifdef INANIWA_SHIFT
extern int inaniwaSide; // set in client_next_game() in sckt.c  B:0 W:1 else:-1
 // after this many move, eval adjust for inaniwa is turned off
#define MAX_INANIWA_NREP  120
static int  calcInaniwaAdjust(tree_t * restrict ptree, int turn);
#endif

#ifdef USE_DIFFEVAL
static int
evaluateScratch( tree_t * restrict ptree, int ply, int turn )
#else
int
evaluate( tree_t * restrict ptree, int ply, int turn )
#endif
{
  int list0[52], list1[52];
  int nlist, score, sq_bk, sq_wk, k0, k1, l0, l1, i, j, sum;
  short scaledScore;

#ifdef INANIWA_SHIFT
  int inaniwaAdjust = 0;
  if (inaniwaSide >= 0 /* fighting against inaniwa */ &&
      root_nrep < MAX_INANIWA_NREP)
    inaniwaAdjust = calcInaniwaAdjust(ptree, turn);
#endif

#ifndef USE_DIFFEVAL
  ptree->neval_called++;

  if ( ptree->stand_pat[ply] != score_bound )
    {
      return (int)ptree->stand_pat[ply];
    }

  if ( ehash_probe( HASH_KEY, HAND_B, &score ) )
    {
      //score                /= FV_SCALE;
      score = MATERIAL + (score - MATERIAL*FV_SCALE) / FV_SCALE;
      score                 = turn ? -score : score;
      ptree->stand_pat[ply] = (short)score;

      return score;
    }
#endif


  score = 0;
  nlist = make_list( ptree, &score, list0, list1 );
  sq_bk = SQ_BKING;
  sq_wk = Inv( SQ_WKING );

  if (DBG_DIFEVAL)
    printf("==== scratch started: kkp_total %d sq_bk %d%d sq_wk %d%d\n",
              score, 9-sq_bk%9, 1+sq_bk/9,9-sq_wk%9,1+sq_wk/9);
  sum = 0;
  for ( i = 0; i < nlist; i++ )
    {
      k0 = list0[i];
      k1 = list1[i];
      for ( j = 0; j <= i; j++ )
	{  int vp, vm;
	  l0 = list0[j];
	  l1 = list1[j];
	  assert( k0 >= l0 && k1 >= l1 );
	  vp  = PcPcOnSq( sq_bk, k0, l0 );
	  vm  = PcPcOnSq( sq_wk, k1, l1 );
          sum += vp - vm;
          if (0 && DBG_DIFEVAL) {
            printf("k0 %s   l0 %s   vp %4d\n", kpptrans(k0), kpptrans2(l0), vp);
            printf("k1 %s   l1 %s   vm %4d\n", kpptrans(k1), kpptrans2(l1), vm);
	  }
	}
    }
  
  score += sum;
  scaledScore = score / FV_SCALE;

  score += MATERIAL * FV_SCALE;
  scaledScore += MATERIAL;

  if (DBG_DIFEVAL) {
    printf("==== scratch ended: sum %d material %d total %d\n",
              sum, MATERIAL, score);
  }

#if ! defined(MINIMUM)
  if ( abs(score) > score_max_eval * FV_SCALE )
    {
      out_warning( "A score at evaluate() is out of bounce." );
    }
#endif

  ehash_store( HASH_KEY, HAND_B, score );  // full value, seen from black

  score = turn ? -score : score;
  scaledScore = turn ? -scaledScore : scaledScore;
#ifdef INANIWA_SHIFT
  scaledScore += inaniwaAdjust;
#endif
  //scaledScore = (short)(score / FV_SCALE);
  ptree->stand_pat[ply] = scaledScore;
#ifdef USE_DIFFEVAL
  ptree->stand_pat_full[ply] = score;  // full value, seen from TURN
#endif

  if (DBG_DIFEVAL) {
    printf("==== scratch : standpat[ply %d] set to %d\n",
              ply, score);
  }

  return scaledScore;

}

#ifdef USE_DIFFEVAL
#include "diffeval.cpp"
#endif


void ehash_clear( void )
{
  int i;

  for ( i = 0; i < EHASH_MASK + 1; i++ ) { ehash_tbl[i] = UINT64_C(0); }
}

#define EH_SCORE_MASK   0x1fffff
#define EH_SCORE_OFFSET 0x100000
#define EH_SCORE_SHIFT  21

static int ehash_probe( uint64_t current_key, unsigned int hand_b,
			int *pscore )
{
  uint64_t hash_word, hash_key;

#ifdef DBG_NO_ATOMIC64
  hash_word = ehash_tbl[ (unsigned int)current_key & EHASH_MASK ];
#else
  ld64b_atomic64(&ehash_tbl[(unsigned int)current_key & EHASH_MASK],&hash_word);
#endif

  current_key ^= (uint64_t)hand_b << EH_SCORE_SHIFT;
  current_key &= ~(uint64_t)EH_SCORE_MASK;

  hash_key  = hash_word;
  hash_key &= ~(uint64_t)EH_SCORE_MASK;

  if ( hash_key != current_key ) { return 0; }

  *pscore = (int)( (unsigned int)hash_word & EH_SCORE_MASK ) - EH_SCORE_OFFSET;

  return 1;
}


static void ehash_store( uint64_t key, unsigned int hand_b, int score )
{
  uint64_t hash_word;

  hash_word  = key;
  hash_word ^= (uint64_t)hand_b << EH_SCORE_SHIFT;
  hash_word &= ~(uint64_t)EH_SCORE_MASK;
  hash_word |= (uint64_t)( score + EH_SCORE_OFFSET );

#ifdef DBG_NO_ATOMIC64
  ehash_tbl[ (unsigned int)key & EHASH_MASK ] = hash_word;
#else
  st64b_atomic64(&ehash_tbl[(unsigned int)key & EHASH_MASK], hash_word);
#endif
}


static int
make_list( const tree_t * restrict ptree, int * restrict pscore,
	   int list0[52], int list1[52] )
{
  bitboard_t bb;
  int list2[34];
  int nlist, sq, n2, i, score, sq_bk0, sq_wk0, sq_bk1, sq_wk1;

  nlist  = 14;
  score  = 0;
  sq_bk0 = SQ_BKING;
  sq_wk0 = SQ_WKING;
  sq_bk1 = Inv(SQ_WKING);
  sq_wk1 = Inv(SQ_BKING);

  list0[ 0] = f_hand_pawn   + I2HandPawn(HAND_B);
  list0[ 1] = e_hand_pawn   + I2HandPawn(HAND_W);
  list0[ 2] = f_hand_lance  + I2HandLance(HAND_B);
  list0[ 3] = e_hand_lance  + I2HandLance(HAND_W);
  list0[ 4] = f_hand_knight + I2HandKnight(HAND_B);
  list0[ 5] = e_hand_knight + I2HandKnight(HAND_W);
  list0[ 6] = f_hand_silver + I2HandSilver(HAND_B);
  list0[ 7] = e_hand_silver + I2HandSilver(HAND_W);
  list0[ 8] = f_hand_gold   + I2HandGold(HAND_B);
  list0[ 9] = e_hand_gold   + I2HandGold(HAND_W);
  list0[10] = f_hand_bishop + I2HandBishop(HAND_B);
  list0[11] = e_hand_bishop + I2HandBishop(HAND_W);
  list0[12] = f_hand_rook   + I2HandRook(HAND_B);
  list0[13] = e_hand_rook   + I2HandRook(HAND_W);

  list1[ 0] = f_hand_pawn   + I2HandPawn(HAND_W);
  list1[ 1] = e_hand_pawn   + I2HandPawn(HAND_B);
  list1[ 2] = f_hand_lance  + I2HandLance(HAND_W);
  list1[ 3] = e_hand_lance  + I2HandLance(HAND_B);
  list1[ 4] = f_hand_knight + I2HandKnight(HAND_W);
  list1[ 5] = e_hand_knight + I2HandKnight(HAND_B);
  list1[ 6] = f_hand_silver + I2HandSilver(HAND_W);
  list1[ 7] = e_hand_silver + I2HandSilver(HAND_B);
  list1[ 8] = f_hand_gold   + I2HandGold(HAND_W);
  list1[ 9] = e_hand_gold   + I2HandGold(HAND_B);
  list1[10] = f_hand_bishop + I2HandBishop(HAND_W);
  list1[11] = e_hand_bishop + I2HandBishop(HAND_B);
  list1[12] = f_hand_rook   + I2HandRook(HAND_W);
  list1[13] = e_hand_rook   + I2HandRook(HAND_B);

  score += kkp[sq_bk0][sq_wk0][ kkp_hand_pawn   + I2HandPawn(HAND_B) ];
  score += kkp[sq_bk0][sq_wk0][ kkp_hand_lance  + I2HandLance(HAND_B) ];
  score += kkp[sq_bk0][sq_wk0][ kkp_hand_knight + I2HandKnight(HAND_B) ];
  score += kkp[sq_bk0][sq_wk0][ kkp_hand_silver + I2HandSilver(HAND_B) ];
  score += kkp[sq_bk0][sq_wk0][ kkp_hand_gold   + I2HandGold(HAND_B) ];
  score += kkp[sq_bk0][sq_wk0][ kkp_hand_bishop + I2HandBishop(HAND_B) ];
  score += kkp[sq_bk0][sq_wk0][ kkp_hand_rook   + I2HandRook(HAND_B) ];

  score -= kkp[sq_bk1][sq_wk1][ kkp_hand_pawn   + I2HandPawn(HAND_W) ];
  score -= kkp[sq_bk1][sq_wk1][ kkp_hand_lance  + I2HandLance(HAND_W) ];
  score -= kkp[sq_bk1][sq_wk1][ kkp_hand_knight + I2HandKnight(HAND_W) ];
  score -= kkp[sq_bk1][sq_wk1][ kkp_hand_silver + I2HandSilver(HAND_W) ];
  score -= kkp[sq_bk1][sq_wk1][ kkp_hand_gold   + I2HandGold(HAND_W) ];
  score -= kkp[sq_bk1][sq_wk1][ kkp_hand_bishop + I2HandBishop(HAND_W) ];
  score -= kkp[sq_bk1][sq_wk1][ kkp_hand_rook   + I2HandRook(HAND_W) ];

  n2 = 0;
  bb = BB_BPAWN;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = f_pawn + sq;
    list2[n2]    = e_pawn + Inv(sq);
    score += kkp[sq_bk0][sq_wk0][ kkp_pawn + sq ];
    nlist += 1;
    n2    += 1;
  }

  bb = BB_WPAWN;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = e_pawn + sq;
    list2[n2]    = f_pawn + Inv(sq);
    score -= kkp[sq_bk1][sq_wk1][ kkp_pawn + Inv(sq) ];
    nlist += 1;
    n2    += 1;
  }
  for ( i = 0; i < n2; i++ ) { list1[nlist-i-1] = list2[i]; }

  n2 = 0;
  bb = BB_BLANCE;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = f_lance + sq;
    list2[n2]    = e_lance + Inv(sq);
    score += kkp[sq_bk0][sq_wk0][ kkp_lance + sq ];
    nlist += 1;
    n2    += 1;
  }

  bb = BB_WLANCE;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = e_lance + sq;
    list2[n2]    = f_lance + Inv(sq);
    score -= kkp[sq_bk1][sq_wk1][ kkp_lance + Inv(sq) ];
    nlist += 1;
    n2    += 1;
  }
  for ( i = 0; i < n2; i++ ) { list1[nlist-i-1] = list2[i]; }


  n2 = 0;
  bb = BB_BKNIGHT;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = f_knight + sq;
    list2[n2]    = e_knight + Inv(sq);
    score += kkp[sq_bk0][sq_wk0][ kkp_knight + sq ];
    nlist += 1;
    n2    += 1;
  }

  bb = BB_WKNIGHT;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = e_knight + sq;
    list2[n2]    = f_knight + Inv(sq);
    score -= kkp[sq_bk1][sq_wk1][ kkp_knight + Inv(sq) ];
    nlist += 1;
    n2    += 1;
  }
  for ( i = 0; i < n2; i++ ) { list1[nlist-i-1] = list2[i]; }


  n2 = 0;
  bb = BB_BSILVER;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = f_silver + sq;
    list2[n2]    = e_silver + Inv(sq);
    score += kkp[sq_bk0][sq_wk0][ kkp_silver + sq ];
    nlist += 1;
    n2    += 1;
  }

  bb = BB_WSILVER;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = e_silver + sq;
    list2[n2]    = f_silver + Inv(sq);
    score -= kkp[sq_bk1][sq_wk1][ kkp_silver + Inv(sq) ];
    nlist += 1;
    n2    += 1;
  }
  for ( i = 0; i < n2; i++ ) { list1[nlist-i-1] = list2[i]; }


  n2 = 0;
  bb = BB_BTGOLD;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = f_gold + sq;
    list2[n2]    = e_gold + Inv(sq);
    score += kkp[sq_bk0][sq_wk0][ kkp_gold + sq ];
    nlist += 1;
    n2    += 1;
  }

  bb = BB_WTGOLD;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = e_gold + sq;
    list2[n2]    = f_gold + Inv(sq);
    score -= kkp[sq_bk1][sq_wk1][ kkp_gold + Inv(sq) ];
    nlist += 1;
    n2    += 1;
  }
  for ( i = 0; i < n2; i++ ) { list1[nlist-i-1] = list2[i]; }


  n2 = 0;
  bb = BB_BBISHOP;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = f_bishop + sq;
    list2[n2]    = e_bishop + Inv(sq);
    score += kkp[sq_bk0][sq_wk0][ kkp_bishop + sq ];
    nlist += 1;
    n2    += 1;
  }

  bb = BB_WBISHOP;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = e_bishop + sq;
    list2[n2]    = f_bishop + Inv(sq);
    score -= kkp[sq_bk1][sq_wk1][ kkp_bishop + Inv(sq) ];
    nlist += 1;
    n2    += 1;
  }
  for ( i = 0; i < n2; i++ ) { list1[nlist-i-1] = list2[i]; }


  n2 = 0;
  bb = BB_BHORSE;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = f_horse + sq;
    list2[n2]    = e_horse + Inv(sq);
    score += kkp[sq_bk0][sq_wk0][ kkp_horse + sq ];
    nlist += 1;
    n2    += 1;
  }

  bb = BB_WHORSE;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = e_horse + sq;
    list2[n2]    = f_horse + Inv(sq);
    score -= kkp[sq_bk1][sq_wk1][ kkp_horse + Inv(sq) ];
    nlist += 1;
    n2    += 1;
  }
  for ( i = 0; i < n2; i++ ) { list1[nlist-i-1] = list2[i]; }


  n2 = 0;
  bb = BB_BROOK;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = f_rook + sq;
    list2[n2]    = e_rook + Inv(sq);
    score += kkp[sq_bk0][sq_wk0][ kkp_rook + sq ];
    nlist += 1;
    n2    += 1;
  }

  bb = BB_WROOK;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = e_rook + sq;
    list2[n2]    = f_rook + Inv(sq);
    score -= kkp[sq_bk1][sq_wk1][ kkp_rook + Inv(sq) ];
    nlist += 1;
    n2    += 1;
  }
  for ( i = 0; i < n2; i++ ) { list1[nlist-i-1] = list2[i]; }


  n2 = 0;
  bb = BB_BDRAGON;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = f_dragon + sq;
    list2[n2]    = e_dragon + Inv(sq);
    score += kkp[sq_bk0][sq_wk0][ kkp_dragon + sq ];
    nlist += 1;
    n2    += 1;
  }

  bb = BB_WDRAGON;
  while ( BBToU(bb) ) {
    sq = FirstOne( bb );
    Xor( sq, bb );

    list0[nlist] = e_dragon + sq;
    list2[n2]    = f_dragon + Inv(sq);
    score -= kkp[sq_bk1][sq_wk1][ kkp_dragon + Inv(sq) ];
    nlist += 1;
    n2    += 1;
  }
  for ( i = 0; i < n2; i++ ) { list1[nlist-i-1] = list2[i]; }

  assert( nlist <= 52 );
  *pscore += score;
  return nlist;
}
