﻿using System.IO;
using System.Text;

namespace Music
{
    public class Midi
    {
        const byte MIDI_NOTEOFF    = 0x80;
        const byte MIDI_NOTEON     = 0x90;
        const byte MIDI_POLYPRESS  = 0xA0;
        const byte MIDI_CTRLCHANGE = 0xB0;
        const byte MIDI_PRGMCHANGE = 0xC0;
        const byte MIDI_CHANPRESS  = 0xD0;
        const byte MIDI_PITCHBEND  = 0xE0;
        const byte MUS_NOTEOFF     = 0x00;
        const byte MUS_NOTEON      = 0x10;
        const byte MUS_PITCHBEND   = 0x20;
        const byte MUS_SYSEVENT    = 0x30;
        const byte MUS_CTRLCHANGE  = 0x40;
        const byte MUS_SCOREEND    = 0x60;
        const int MAX_MIDI_SIZE = 128 * 1024;
        public MemoryStream data;
        public string name;
        public int secs;
        byte[] midiHeader = {
            0x4d, 0x54, 0x68, 0x64,     // chunk type ("MThd")
            0, 0, 0, 6,                 // header length (次の3つのフィールドのバイト数で、常に6)
            0, 0,                       // format 0: only one track
            0, 1,                       // yes, there is really only one track
            0, 70,                      // 70 divisions (デルタタイムの単位が4分音符の70分の1)
            0x4d, 0x54, 0x72, 0x6b,     // chunk type ("MTrk")
            0, 0, 0, 0,                 // track length (最後に代入される)
            0,                          // メタイベントであってもデルタタイムが先行しなければいけないと思われる
            // The first event sets the tempo to 500,000 microsec/quarter note (4分音符の長さが0.5秒)
            255, 81, 3,                 // 「Windowsサウンドプログラミング」23.6 メタイベント 参照
            0x07, 0xa1, 0x20,           // 500,000 になるのは確認済み
            0,
            0xB9, 7, 127                // Set the percussion channel to full volume
        };
        byte[] CtrlNumShift = {
            0,                          // 0	program change
            0,                          // 1	bank select
            1,                          // 2	modulation pot
            7,                          // 3	volume
            10,                         // 4	pan pot
            11,                         // 5	expression pot
            91,                         // 6	reverb depth
            93,                         // 7	chorus depth
            64,                         // 8	sustain pedal
            67,                         // 9	soft pedal
            // 以降はMUSではsystem event、MIDIではチャンネルモードメッセージ
            120,                        // 10	all sounds off
            123,                        // 11	all notes off
            126,                        // 12	mono
            127,                        // 13	poly
            121,                        // 14	reset all controllers
        };

        public Midi()
        {
            name = null;
            secs = 0;
            data = null;
        }

        public Midi(string s)
        {
            name = s;
            secs = 0;
            data = null;
        }

        public Midi(string s, MemoryStream ms)
        {
            BinaryReader br;
            BinaryWriter bw;
            char[] id4;
            ushort scoreStart;
            byte musevent = 0, musdat1 = 0;
            byte status, data1, data2;
            bool bTwoData;
            int i;
            byte[] lastVel = new byte[16]; // MUSはplay note(note on)の時にnote volume(velocity)が前と同じ時は省略するが、MIDIはnote onの時は必ずvelocityが続くので前のnote volumeを記憶しておく
            int[] chanMap = new int[16];
            int muschan;
            int midiChan = 0;
            int deltaTime = 0;
            int tics = 0;
            long size;
            long trackLen;

            name = s;
            br = new BinaryReader(ms, Encoding.ASCII);
            ms.Seek(0, SeekOrigin.Begin);
            id4 = br.ReadChars(4);
            if (!(id4[0] == 'M' && id4[1] == 'U' && id4[2] == 'S'))
            {
                data = null;
                return;
            }
            data = new MemoryStream(MAX_MIDI_SIZE);
            bw = new BinaryWriter(data);
            data.Seek(0, SeekOrigin.Begin);
            bw.Write(midiHeader);
            ms.Seek(6, SeekOrigin.Begin);
            scoreStart = br.ReadUInt16();
            ms.Seek((long)scoreStart, SeekOrigin.Begin);
            for (i=0 ; i<16 ; i++) lastVel[i] = 64;
            for (i=0 ; i<15 ; i++) chanMap[i] = -1;
            chanMap[15] = 9; // percussion用のチャンネルは、GeneralMIDIが10番目のチャンネル、MUSが16番目のチャンネル
            while ((musevent & 0x70) != MUS_SCOREEND)
            {
                bTwoData = true;
                musevent = br.ReadByte();
                if ((musevent & 0x70) != MUS_SCOREEND)
                    musdat1 = br.ReadByte();
                muschan = musevent & 15; // 下位4ビットだけを取り出す
                if (chanMap[muschan] < 0)
                {
                    // This is the first time this channel has been used,
                    // so sets its volume to 127.
                    bw.Write((byte)0); // デルタタイム
                    bw.Write((byte)(0xB0 | midiChan)); // ステータスバイト
                    bw.Write((byte)7); // data1
                    bw.Write((byte)127); // data2
                    chanMap[muschan] = midiChan++; // (MUSでのチャンネルが何番であれ)現れた順に0からMIDIのチャンネルに変換する
                    if (midiChan == 9) // 9は迂回する
                        midiChan++;
                }
                status = (byte)chanMap[muschan];
                switch (musevent & 0x70) // 0x70は2進数で01110000
                {
                    case MUS_NOTEOFF:
                        status |= MIDI_NOTEOFF;
                        data1 = musdat1;
                        data2 = 64; // オフベロシティ。MIDIでもあまり使われていない
                        break;

                    case MUS_NOTEON:
                        status |= MIDI_NOTEON;
                        data1 = (byte)(musdat1 & 127);
                        if ((musdat1 & 128) != 0) // 'Vol'ビットが立っている
                            lastVel[muschan] = br.ReadByte(); // note volume(velocity)を更新
                        data2 = lastVel[muschan];
                        break;

                    case MUS_PITCHBEND: // MUSのpitch wheel valueは8bit値なので、MIDIでの2バイト形式(「Windowsサウンドプログラミング」13.3 参照)に変換する
                        status |= MIDI_PITCHBEND; 
                        data1 = (byte)((musdat1 & 1) << 6); // 下位バイト
                        data2 = (byte)((uint)musdat1 >> 1); // 上位バイト
                        break;

                    case MUS_SYSEVENT:
                        status |= MIDI_CTRLCHANGE;
                        data1 = CtrlNumShift[musdat1];
                        data2 = 0; // mono mode(musdat1が12)の時の3バイト目の値については、「コンプリートMIDIブック」P108 参照
                        break;

                    case MUS_CTRLCHANGE:
                        if (musdat1 == 0) // program(patch,instrument) change (唯一のMIDIのデータバイトが1つの場合)
                        {
                            status |= MIDI_PRGMCHANGE;
                            data1 = br.ReadByte();
                            bTwoData = false;
                            data2 = 0; // ビルドエラー抑制のためだけ
                        }
                        else
                        {
                            status |= MIDI_CTRLCHANGE;
                            data1 = CtrlNumShift[musdat1];
                            data2 = br.ReadByte();
                        }
                        break;

                    case MUS_SCOREEND:
                        status = 0xFF;  // 唯一のメタイベント(0xFFで始まる)
                        data1 = 0x2F;   // End of Track(「Windowsサウンドプログラミング」P236)
                        data2 = 0x00;
                        break;

                    default:
                        data = null;
                        return;
                }
                WriteVarLen(bw, deltaTime); // MIDIの場合は必ずデルタタイムを入れる
                bw.Write(status);
                bw.Write(data1);
                if (bTwoData)
                    bw.Write(data2);
                if ((musevent & 128) != 0) // 同時間のイベントの中の最後なので、次にデルタタイムが来る
                    ReadVarLen(br, out deltaTime);
                else
                    deltaTime = 0;
                tics += deltaTime;
            }
            size = data.Position;
            trackLen = size - 22;
            data.Seek(18, SeekOrigin.Begin);
            bw.Write((byte)((trackLen >> 24) & 255));
            bw.Write((byte)((trackLen >> 16) & 255));
            bw.Write((byte)((trackLen >> 8) & 255));
            bw.Write((byte)(trackLen & 255));
            data.SetLength(size);
            secs = tics / 140;
        }

        void WriteVarLen(BinaryWriter bw, int time)
        {
            long buffer;

            buffer = time & 0x7f;
            while ((time >>= 7) > 0)
                buffer = (buffer << 8) + 0x80 + (time & 0x7f);
            for (;;)
            {
                bw.Write((byte)(buffer & 0xff));
                if ((buffer & 0x80) != 0)
                    buffer >>= 8;
                else
                    break;
            }
        }

        void ReadVarLen(BinaryReader br, out int time_out)
        {
            int time = 0;
            byte t;

            do {
                t = br.ReadByte();
                time = (time << 7) | (t & 127);
            } while ((t & 128) != 0);
            time_out = time;
        }
    }
}