#include <stdlib.h>
#include <math.h>
#include <malloc.h>
#include "r_state.h"
#include "i_system.h"
#include "z_zone.h"
#include "m_bbox.h"

typedef struct
{
	int     count;
	int     size;
	byte*   data;

} storage_t;


void initStorage(storage_t* storage, int size)
{
    storage->count = 0;
    storage->size = size;
    storage->data = malloc(size);
}

void addElement(storage_t* storage, void* element)
{
    int     count, size;
    byte*   data;

    count = storage->count;
    size = storage->size;
    data = storage->data;

    data = realloc(data, size*(count+1));
    memcpy(data+(size*count), element, size);
    storage->data = data;
    storage->count = ++count;
}

int getCount(storage_t* storage)
{
    return storage->count;
}

void* elementAt(storage_t* storage, int i)
{
    int     count, size;
    byte*   data;

    count = storage->count;
    size = storage->size;
    data = storage->data;

    if (i >= count || i < 0)
        return NULL;

    return data+(size*i);
}

void freeStorage(storage_t* storage)
{
    free(storage->data);
    storage->count = 0;
}


typedef struct
{
    fvertex_t   v1;
    fvertex_t   v2;

} fseg_t;

typedef struct
{
    fvertex_t   pt;
    float       dx, dy;

} unitvector_t;

typedef struct
{
    int     segnum;
    int     end;    // 0 = v1, 1 = v2

} segend_t;

typedef struct
{
    int         type1;  // 0 = segend, 1 = fvertex
    segend_t    segend1;
    fvertex_t   v1;

    int         type2;
    segend_t    segend2;
    fvertex_t   v2;

} invisseg_t; // invisible seg

//#define TOLERANCE   2.0f
float TOLERANCE;

unitvector_t divvector; // @segendcmp()Ŏg߃O[oϐɂĂ


float InnerProduct(float x1, float y1, float x2, float y2)
{
    return x1*x2 + y1*y2;
}

float OuterProduct(float x1, float y1, float x2, float y2)
{
    return x1*y2 - x2*y1;
}

int segendcmp(const void *elem1, const void *elem2)
{
    segend_t*   segend1 = (segend_t *)elem1;
    segend_t*   segend2 = (segend_t *)elem2;
    int         index;
    float       x2, y2;
    float       val1, val2;

    index = segend1->segnum;
    if (!segend1->end)
    {
        x2 = FIXEDTOFLOAT(segs[index].v1->x);
        y2 = FIXEDTOFLOAT(segs[index].v1->y);
    } else
    {
        x2 = FIXEDTOFLOAT(segs[index].v2->x);
        y2 = FIXEDTOFLOAT(segs[index].v2->y);
    }
    x2 -= divvector.pt.x;
    y2 -= divvector.pt.y;
    val1 = InnerProduct(divvector.dx, divvector.dy, x2, y2);

    index = segend2->segnum;
    if (!segend2->end)
    {
        x2 = FIXEDTOFLOAT(segs[index].v1->x);
        y2 = FIXEDTOFLOAT(segs[index].v1->y);
    } else
    {
        x2 = FIXEDTOFLOAT(segs[index].v2->x);
        y2 = FIXEDTOFLOAT(segs[index].v2->y);
    }
    x2 -= divvector.pt.x;
    y2 -= divvector.pt.y;
    val2 = InnerProduct(divvector.dx, divvector.dy, x2, y2);

    if (val1 - val2 > TOLERANCE)
        return 1;
    if (val2 - val1 > TOLERANCE)
        return -1;
    return 0;
}

void UnitvectorFromNode(unitvector_t* unitvector, node_t* node)
{
    float   dx, dy;
    float   len;

    unitvector->pt.x = FIXEDTOFLOAT(node->x);
    unitvector->pt.y = FIXEDTOFLOAT(node->y);

    dx = FIXEDTOFLOAT(node->dx);
    dy = FIXEDTOFLOAT(node->dy);
    len = (float)sqrt(dx*dx + dy*dy);
    unitvector->dx = dx/len;
    unitvector->dy = dy/len;
}

void UnitvectorFromSeg(unitvector_t* unitvector, int segnum)
{
    vertex_t    *v1, *v2;
    node_t      node;

    v1 = segs[segnum].v1;
    v2 = segs[segnum].v2;
    node.x = v1->x;
    node.y = v1->y;
    node.dx = v2->x - v1->x;
    node.dy = v2->y - v1->y;
    UnitvectorFromNode(unitvector, &node);
}

void UnitvectorFromFSeg(unitvector_t* unitvector, fseg_t* fseg)
{
    float   dx, dy;
    float   len;

    unitvector->pt = fseg->v1;

    dx = fseg->v2.x - fseg->v1.x;
    dy = fseg->v2.y - fseg->v1.y;
    len = (float)sqrt(dx*dx + dy*dy);
    unitvector->dx = dx/len;
    unitvector->dy = dy/len;
}

boolean ClosePoints(fvertex_t* p1, fvertex_t* p2)
{
    float   dx, dy;
    //float   val;

    dx = p2->x - p1->x;
    dy = p2->y - p1->y;
/*
    val = (float)sqrt(dx*dx + dy*dy);

    if (val < TOLERANCE)
        return true;
*/
    if ((dx > -TOLERANCE && dx < TOLERANCE) && (dy > -TOLERANCE && dy < TOLERANCE))
        return true;

    return false;
}

boolean ClosePoints2(vertex_t* v1, vertex_t* v2)
{
    fvertex_t   p1, p2;

    p1.x = FIXEDTOFLOAT(v1->x);
    p1.y = FIXEDTOFLOAT(v1->y);
    p2.x = FIXEDTOFLOAT(v2->x);
    p2.y = FIXEDTOFLOAT(v2->y);

    return ClosePoints(&p1, &p2);
}

int VertexOnSide(vertex_t* vertex, unitvector_t* unitvector)
{
    float   val;

    val = OuterProduct(unitvector->dx, unitvector->dy,
                       FIXEDTOFLOAT(vertex->x) - unitvector->pt.x,
                       FIXEDTOFLOAT(vertex->y) - unitvector->pt.y);

    if (val < -TOLERANCE)
        return 0;
    if (val > TOLERANCE)
        return 1;
    return -1;
}

int SegOnDivSide(seg_t* seg)
{
    int     ret1, ret2;

    ret1 = VertexOnSide(seg->v1, &divvector);
    ret2 = VertexOnSide(seg->v2, &divvector);

    if (ret1 == 0 && ret2 == 0)
        return 0;
    if (ret1 == 1 && ret2 == 1)
        return 1;
    if (ret1 == -1)
    {
        if (ret2 == 0)
            return 0;
        else if (ret2 == 1)
            return 1;
    }
    if (ret2 == -1)
    {
        if (ret1 == 0)
            return 0;
        else if (ret1 == 1)
            return 1;
    }
    return -1;
}

boolean CheckConcursegs(storage_t* concursegs, vertex_t* vertex)
{
    int             c, i, j;
    int             *segnum, *segnum1, *segnum2;
    int*            check;
    angle_t         a, b;
    unitvector_t    segvector;
    int             side, backside = 0;

    c = getCount(concursegs);

    for (i=0 ; i<c ; i++)
    {
        segnum = elementAt(concursegs, i);
        if (ClosePoints2(vertex, segs[*segnum].v1) || ClosePoints2(vertex, segs[*segnum].v2))
            return false;
    }

    // @double sided linedef̑΂ɂȂĂsegr
    check = (int *)alloca(c*sizeof(int));
    memset(check, 0, c*sizeof(int));
    for (i=0 ; i<c ; i++)
    {
        for (j=i+1 ; j<c ; j++) // @1(n-1)܂ł̐𑫂((n-1)*((n-1)+1))/2ƁAn̗vf2̗vfogݍ킹͈̌v
        {
            segnum1 = elementAt(concursegs, i);
            segnum2 = elementAt(concursegs, j);
            if (ClosePoints2(segs[*segnum1].v1, segs[*segnum2].v2) || ClosePoints2(segs[*segnum1].v2, segs[*segnum2].v1))
            {
                a = segs[*segnum1].angle;
                b = segs[*segnum2].angle + ANG180;
                if (abs((int)(a-b)) < (ANG45/45))
                {
                    if (SegOnDivSide(&segs[*segnum1]) == SegOnDivSide(&segs[*segnum2]))
                    {
                        check[i] = 1;
                        check[j] = 1;
                    }
                }
            }
        }
    }

    for (i=0 ; i<c ; i++)
    {
        if (!check[i])
        {
            segnum = elementAt(concursegs, i);
            UnitvectorFromSeg(&segvector, *segnum);
            side = VertexOnSide(vertex, &segvector);
            if (side == 1 || side == -1)
                backside++;
        }
    }
    if (backside >= 2)
        return false;

    return true;
}

void CutInvisseg(invisseg_t* invisseg, unitvector_t* unitvector, invisseg_t* newinvisseg, invisseg_t* divends)
{
    int         segnum;
    float       o1x, o1y, o2x, o2y;
    float       d1x, d1y, d1px, d1py, d2x, d2y, d2px, d2py;
    float       num, den, s, t;
    float       newx, newy;
    fvertex_t   fv1, fv2;

    o1x = unitvector->pt.x;
    o1y = unitvector->pt.y;

    d1x = unitvector->dx;
    d1y = unitvector->dy;

    if (invisseg->type1)
    {
        o2x = invisseg->v1.x;
        o2y = invisseg->v1.y;
    } else
    {
        segnum = invisseg->segend1.segnum;
        if (invisseg->segend1.end)
        {
            o2x = FIXEDTOFLOAT(segs[segnum].v2->x);
            o2y = FIXEDTOFLOAT(segs[segnum].v2->y);
        } else
        {
            o2x = FIXEDTOFLOAT(segs[segnum].v1->x);
            o2y = FIXEDTOFLOAT(segs[segnum].v1->y);
        }
    }

    if (invisseg->type2)
    {
        d2x = invisseg->v2.x;
        d2y = invisseg->v2.y;
    } else
    {
        segnum = invisseg->segend2.segnum;
        if (invisseg->segend2.end)
        {
            d2x = FIXEDTOFLOAT(segs[segnum].v2->x);
            d2y = FIXEDTOFLOAT(segs[segnum].v2->y);
        } else
        {
            d2x = FIXEDTOFLOAT(segs[segnum].v1->x);
            d2y = FIXEDTOFLOAT(segs[segnum].v1->y);
        }
    }
    d2x -= o2x;
    d2y -= o2y;
    d2px = -d2y;
    d2py = d2x;

    // Real-Time Rendering SE p617 (13.46)
    num = InnerProduct(o2x - o1x, o2y - o1y, d2px, d2py);
    den = InnerProduct(d1x, d1y, d2px, d2py);
    s = num / den;

    newx = o1x + s*d1x;
    newy = o1y + s*d1y;

    *newinvisseg = *invisseg;
    if (OuterProduct(d1x, d1y, o2x - o1x, o2y - o1y) < 0)
    {
        invisseg->type2 = 1;
        newinvisseg->type1 = 1;
        newinvisseg->v1.x = invisseg->v2.x = newx;
        newinvisseg->v1.y = invisseg->v2.y = newy;
    } else
    {
        invisseg->type1 = 1;
        newinvisseg->type2 = 1;
        newinvisseg->v2.x = invisseg->v1.x = newx;
        newinvisseg->v2.y = invisseg->v1.y = newy;
    }

    if (s > 0)
    {
        divends->type2 = 1;
        divends->v2.x = newx;
        divends->v2.y = newy;
    } else
    {
        divends->type1 = 1;
        divends->v1.x = newx;
        divends->v1.y = newy;
    }

    // @ȉ͊mFׂ̈̂
    d1px = -d1y;
    d1py = d1x;
    num = InnerProduct(o1x - o2x, o1y - o2y, d1px, d1py);
    den = InnerProduct(d2x, d2y, d1px, d1py);
    t = num / den;

    if (t < 0.0f || t > 1.0f)
        I_Debug("CutInvisseg: the value of t out of range\n");

    fv1.x = newx;
    fv1.y = newy;
    fv2.x = o2x + t*d2x;
    fv2.y = o2y + t*d2y;

    if (!ClosePoints(&fv1, &fv2))
        I_Debug("CutInvisseg: the wrong intersection point\n");
}

int InvissegOnSide(invisseg_t* invisseg, unitvector_t* unitvector, invisseg_t* divends)
{
    float   v1x, v1y, v2x, v2y;
    float   v1dx, v1dy, v2dx, v2dy;
    int     segnum;
    float   val1, val2;

    if (invisseg->type1)
    {
        v1x = invisseg->v1.x;
        v1y = invisseg->v1.y;
    } else
    {
        segnum = invisseg->segend1.segnum;
        if (invisseg->segend1.end)
        {
            v1x = FIXEDTOFLOAT(segs[segnum].v2->x);
            v1y = FIXEDTOFLOAT(segs[segnum].v2->y);
        } else
        {
            v1x = FIXEDTOFLOAT(segs[segnum].v1->x);
            v1y = FIXEDTOFLOAT(segs[segnum].v1->y);
        }
    }
    v1dx = v1x - unitvector->pt.x;
    v1dy = v1y - unitvector->pt.y;
    val1 = OuterProduct(unitvector->dx, unitvector->dy, v1dx, v1dy);

    if (invisseg->type2)
    {
        v2x = invisseg->v2.x;
        v2y = invisseg->v2.y;
    } else
    {
        segnum = invisseg->segend2.segnum;
        if (invisseg->segend2.end)
        {
            v2x = FIXEDTOFLOAT(segs[segnum].v2->x);
            v2y = FIXEDTOFLOAT(segs[segnum].v2->y);
        } else
        {
            v2x = FIXEDTOFLOAT(segs[segnum].v1->x);
            v2y = FIXEDTOFLOAT(segs[segnum].v1->y);
        }
    }
    v2dx = v2x - unitvector->pt.x;
    v2dy = v2y - unitvector->pt.y;
    val2 = OuterProduct(unitvector->dx, unitvector->dy, v2dx, v2dy);

    if (val1 < -TOLERANCE && val2 < -TOLERANCE)
        return 0;
    if (val1 > TOLERANCE && val2 >TOLERANCE)
        return 1;
    if (val1 > -TOLERANCE && val1 < TOLERANCE)
    {
        if (InnerProduct(unitvector->dx, unitvector->dy, v1dx, v1dy) > TOLERANCE)
        {
            divends->type2 = 1;
            divends->v2.x = v1x;
            divends->v2.y = v1y;
        } else
        {
            divends->type1 = 1;
            divends->v1.x = v1x;
            divends->v1.y = v1y;
        }
        if (val2 < -TOLERANCE)
            return 0;
        else if (val2 > TOLERANCE)
            return 1;
    }
    if (val2 > -TOLERANCE && val2 < TOLERANCE)
    {
        if (InnerProduct(unitvector->dx, unitvector->dy, v2dx, v2dy) > TOLERANCE)
        {
            divends->type2 = 1;
            divends->v2.x = v2x;
            divends->v2.y = v2y;
        } else
        {
            divends->type1 = 1;
            divends->v1.x = v2x;
            divends->v1.y = v2y;
        }
        if (val1 < -TOLERANCE)
            return 0;
        else if (val1 > TOLERANCE)
            return 1;
    }
    if ((val1 < -TOLERANCE && val2 > TOLERANCE) || (val1 > TOLERANCE && val2 < -TOLERANCE))
        return -1;

    // @divline߂鎞ɁAȂꍇ́AclineSĕNȂȂ܂Œׂ̂(buildbsp.mQ)A
    // @divlineȑOdivlineƏdȂ邱Ƃ͖Ǝv
    I_Debug("InvissegOnSide: an invisseg is aligned with the divline\n");
    if (InnerProduct(unitvector->dx, unitvector->dy, v2x-v1x, v2y-v1y) < 0)
    {
        I_Debug("\topposit direction\n");
        return 1;
    }
    return 0;
}

int FSegOnSide(fseg_t* fseg, unitvector_t* unitvector)
{
    float   x2, y2;
    float   val;

    x2 = fseg->v1.x - unitvector->pt.x;
    y2 = fseg->v1.y - unitvector->pt.y;
    val = OuterProduct(unitvector->dx, unitvector->dy, x2, y2);
    if (val > TOLERANCE)
        return 1;

    x2 = fseg->v2.x - unitvector->pt.x;
    y2 = fseg->v2.y - unitvector->pt.y;
    val = OuterProduct(unitvector->dx, unitvector->dy, x2, y2);
    if (val > TOLERANCE)
        return 1;

    return 0;
}

void ConnectSSecSegs(int ssecnum, storage_t* realsegs, storage_t* invissegs)
{
    subsector_t*    sub;
    int             c, i, j, k;
    int*            segnum_p;
    int             numlines, numinvissegs, numfsegs;
    fseg_t*         fsegs;
    fvertex_t*      vertices;
    seg_t*          seg;
    fvertex_t        v, startv;
    invisseg_t*     invisseg;
    int             segnum;
    unitvector_t    unitvector;
    int             firsttime = 1;

    sub = &subsectors[ssecnum];
    c = getCount(realsegs);

    for (i=0 ; i<c ; i++)
    {
        segnum_p = elementAt(realsegs, i);
        if (&segs[*segnum_p] < &segs[sub->firstline] || &segs[*segnum_p] > (&segs[sub->firstline] + sub->numlines-1))
        {
            I_Debug("ConnectSSecSegs: the segs do not match\n");
            return;
        }
    }

    numlines = sub->numlines;
    numinvissegs = getCount(invissegs);
    //I_Debug("ConnectSSecSegs: numlines = %d numinvissegs = %d\n", numlines, numinvissegs);
    numfsegs = numlines + numinvissegs;

    if (numfsegs < 3)
    {
        I_Debug("ConnectSSecSegs: found a subsector that has segs less than 3\n");
        //return;
    }

    fsegs = calloc(numfsegs, sizeof(fseg_t));
    seg = &segs[sub->firstline];
    for (i=0 ; i< numlines ; i++, seg++)
    {
        fsegs[i].v1.x = FIXEDTOFLOAT(seg->v1->x);
        fsegs[i].v1.y = FIXEDTOFLOAT(seg->v1->y);
        fsegs[i].v2.x = FIXEDTOFLOAT(seg->v2->x);
        fsegs[i].v2.y = FIXEDTOFLOAT(seg->v2->y);
    }
    for (j=0 ; j<numinvissegs ; j++, i++)
    {
        invisseg = elementAt(invissegs, j);

        if (invisseg->type1)
            fsegs[i].v1 = invisseg->v1;
        else
        {
            segnum = invisseg->segend1.segnum;
            if (invisseg->segend1.end)
            {
                fsegs[i].v1.x = FIXEDTOFLOAT(segs[segnum].v2->x);
                fsegs[i].v1.y = FIXEDTOFLOAT(segs[segnum].v2->y);
            }
            else
            {
                fsegs[i].v1.x = FIXEDTOFLOAT(segs[segnum].v1->x);
                fsegs[i].v1.y = FIXEDTOFLOAT(segs[segnum].v1->y);
            }
        }

        if (invisseg->type2)
            fsegs[i].v2 = invisseg->v2;
        else
        {
            segnum = invisseg->segend2.segnum;
            if (invisseg->segend2.end)
            {
                fsegs[i].v2.x = FIXEDTOFLOAT(segs[segnum].v2->x);
                fsegs[i].v2.y = FIXEDTOFLOAT(segs[segnum].v2->y);
            }
            else
            {
                fsegs[i].v2.x = FIXEDTOFLOAT(segs[segnum].v1->x);
                fsegs[i].v2.y = FIXEDTOFLOAT(segs[segnum].v1->y);
            }
        }
/*
        // invisseg length test
        {
            float   dx, dy, len;

            dx = fsegs[i].v2.x - fsegs[i].v1.x;
            dy = fsegs[i].v2.y - fsegs[i].v1.y;
            len = (float)sqrt(dx*dx + dy*dy);
            if (len < 5.0f)
                I_Debug("ConnectSSecSegs: an invisseg with the length less than 5 found\n");
        }*/
    }

    // convexicity test
    for (i=0 ; i</*numlines*/numfsegs ; i++)
    {
        UnitvectorFromFSeg(&unitvector, &fsegs[i]);

        for (j=0 ; j</*numlines*/numfsegs ; j++)
        {
            if (j == i)
                continue;

            if (FSegOnSide(&fsegs[j], &unitvector))
            {
                if (firsttime)
                {
                    I_Debug("ConnectSSecSegs: subsector is not convex\n");
                    firsttime = 0;
                }
            }
        }
    }

    sub->poly.vertices = vertices = Z_Malloc((numfsegs+1)*sizeof(fvertex_t), PU_LEVEL, 0);
    k = 0;
retry:
    i = 0;
    vertices[i++] = startv = fsegs[k].v1;
    v = fsegs[k].v2;
    while (!ClosePoints(&v, &startv) && i < numfsegs+1)
    {
        for (j=0 ; j<numfsegs ; j++)
            if (ClosePoints(&v, &fsegs[j].v1))
            {
                vertices[i++] = v;
                v = fsegs[j].v2;
                break;
            }

        if (j == numfsegs)
        {
            if (i < numfsegs && ++k < numfsegs)
                goto retry;
            I_Debug("ConnectSSecSegs: subsector is not closed\n");
            vertices[i++] = v;
            break;
        }
    }
    sub->poly.numvertices = i;
/*
    if (i > numfsegs)
        I_Debug("ConnectSSecSegs: the number of a subsector vertices exceeded the number of the segs\n");
    else if (i < numfsegs)
        I_Debug("ConnectSSecSegs: the number of a subsector vertices fell short of the number of the segs\n");
*/
    free(fsegs);
}

void DivideNodeSegs(int nodenum, storage_t* realsegs, storage_t* invissegs)
{
    node_t*	    node;
    storage_t   frontrealsegs, backrealsegs;
    int         c, i, cut;
    int*        index;
    int		    s1, s2, side;
    segend_t    segend;
    storage_t   cosegends;  // colinear seg ends
    float       val;
    storage_t   frontinvissegs, backinvissegs;
    invisseg_t* invisseg_p;
    invisseg_t  newinvisseg;
    invisseg_t  divends;    // @{invisseg_t̎gƈႤƂɒ
    storage_t   concursegs; // concurrent segs
    invisseg_t  invisseg;
    segend_t    *segend1, *segend2;
    vertex_t    *vertex1, *vertex2;
  
    if (nodenum & NF_SUBSECTOR)
    {
        ConnectSSecSegs(nodenum&(~NF_SUBSECTOR), realsegs, invissegs);
        return;
    }

    node = &nodes[nodenum];
    UnitvectorFromNode(&divvector, node);

    initStorage(&frontrealsegs, sizeof(int));
    initStorage(&backrealsegs, sizeof(int));
    initStorage(&cosegends, sizeof(segend_t));
    c = getCount(realsegs);
    for (i=0 ; i<c ; i++)
    {
        index = elementAt(realsegs, i);
        s1 = VertexOnSide(segs[*index].v1, &divvector);
        s2 = VertexOnSide(segs[*index].v2, &divvector);
        if (s1 == 0 && s2 == 0)
            addElement(&frontrealsegs, index);
        else if (s1 == 1 && s2 == 1)
            addElement(&backrealsegs, index);
        else if (s1 == -1 && s2 == -1)
        {
            val = InnerProduct(divvector.dx, divvector.dy, 
                               FIXEDTOFLOAT(segs[*index].v2->x - segs[*index].v1->x),
                               FIXEDTOFLOAT(segs[*index].v2->y - segs[*index].v1->y));
            if (val > 0)
                addElement(&frontrealsegs, index);
            else
                addElement(&backrealsegs, index);
            segend.segnum = *index;
            segend.end = 0;
            addElement(&cosegends, &segend);
            segend.end = 1;
            addElement(&cosegends, &segend);
        }
        else if (s1 == -1)
        {
            if (s2 == 0)
                addElement(&frontrealsegs, index);
            else
                addElement(&backrealsegs, index);
            segend.segnum = *index;
            segend.end = 0;
            addElement(&cosegends, &segend);
        }
        else if (s2 == -1)
        {
            if (s1 == 0)
                addElement(&frontrealsegs, index);
            else
                addElement(&backrealsegs, index);
            segend.segnum = *index;
            segend.end = 1;
            addElement(&cosegends, &segend);
        }
        else
        {
            float   x2, y2, val1, val2;

            x2 = FIXEDTOFLOAT(segs[*index].v1->x) - divvector.pt.x;
            y2 = FIXEDTOFLOAT(segs[*index].v1->y) - divvector.pt.y;
            val1 = OuterProduct(divvector.dx, divvector.dy, x2, y2);
            x2 = FIXEDTOFLOAT(segs[*index].v2->x) - divvector.pt.x;
            y2 = FIXEDTOFLOAT(segs[*index].v2->y) - divvector.pt.y;
            val2 = OuterProduct(divvector.dx, divvector.dy, x2, y2);
            if (fabs(val1) > fabs(val2))
            {
                if (s1 == 0)
                    addElement(&frontrealsegs, index);
                else
                    addElement(&backrealsegs, index);
                segend.segnum = *index;
                segend.end = 1;
                addElement(&cosegends, &segend);
            } else
            {
                if (s2 == 0)
                    addElement(&frontrealsegs, index);
                else
                    addElement(&backrealsegs, index);
                segend.segnum = *index;
                segend.end = 0;
                addElement(&cosegends, &segend);
            }

            I_Debug("DivideNodeSegs: a seg straddled the divline\n");
        }
    }

    initStorage(&frontinvissegs, sizeof(invisseg_t));
    initStorage(&backinvissegs, sizeof(invisseg_t));
    memset(&divends, 0, sizeof(invisseg_t));
    cut = 0;
    c = getCount(invissegs);
    for (i=0 ; i<c ; i++)
    {
        invisseg_p = elementAt(invissegs, i);
        side = InvissegOnSide(invisseg_p, &divvector, &divends);
        switch (side)
        {
        case 0:
            addElement(&frontinvissegs, invisseg_p);
            break;
        case 1:
            addElement(&backinvissegs, invisseg_p);
            break;
        case -1:
            CutInvisseg(invisseg_p, &divvector, &newinvisseg, &divends);
            addElement(&frontinvissegs, invisseg_p);
            addElement(&backinvissegs, &newinvisseg);
            cut++;
            break;
        }
    }
    if (cut > 2)
        I_Debug("DivideNodeSegs: cut invissegs more than twice\n");
       
    c = getCount(&cosegends);
    if (!c) // @divline?
    {
        I_Debug("DivideNodeSegs: no cosegends found\n");
        initStorage(&concursegs, sizeof(int));
        goto childnodes;
        //return;
    }
    qsort(elementAt(&cosegends, 0), c, sizeof(segend_t), segendcmp);

    initStorage(&concursegs, sizeof(int));
    segend1 = elementAt(&cosegends, 0);
    if (!segend1->end)
        vertex1 = segs[segend1->segnum].v1;
    else
        vertex1 = segs[segend1->segnum].v2;
    addElement(&concursegs, &segend1->segnum);

    if (divends.type1)  // @invissegvoid spaceɂ͖̂ŁAKZN^[n܂
    {
        fvertex_t   fv;

        fv.x = FIXEDTOFLOAT(vertex1->x);
        fv.y = FIXEDTOFLOAT(vertex1->y);
        if (!ClosePoints(&fv, &divends.v1))
        {
        invisseg.type1 = 1;
        invisseg.v1 = divends.v1;
        invisseg.type2 = 0;
        invisseg.segend2 = *segend1;
        addElement(&frontinvissegs, &invisseg);
        invisseg.type2 = 1;
        invisseg.v2 = divends.v1;
        invisseg.type1 = 0;
        invisseg.segend1 = *segend1;
        addElement(&backinvissegs, &invisseg);
        }
    }

    for (i=1 ; i<c ; i++)
    {
        segend2 = elementAt(&cosegends, i);
        if (!segend2->end)
            vertex2 = segs[segend2->segnum].v1;
        else
            vertex2 = segs[segend2->segnum].v2;

        if (ClosePoints2(vertex1, vertex2))
        {
            addElement(&concursegs, &segend2->segnum);
            continue;
        }

        if (CheckConcursegs(&concursegs, vertex2))
        {
            invisseg.type1 = 0;
            invisseg.segend1 = *segend1;
            invisseg.type2 = 0;
            invisseg.segend2 = *segend2;
            addElement(&frontinvissegs, &invisseg);
            invisseg.type1 = 0;
            invisseg.segend1 = *segend2;
            invisseg.type2 = 0;
            invisseg.segend2 = *segend1;
            addElement(&backinvissegs, &invisseg);
        }

        vertex1 = vertex2;
        segend1 = segend2;
        freeStorage(&concursegs);
        initStorage(&concursegs, sizeof(int));
        addElement(&concursegs, &segend1->segnum);
    }

    if (divends.type2)
    {
        fvertex_t   fv;

        fv.x = FIXEDTOFLOAT(vertex1->x);
        fv.y = FIXEDTOFLOAT(vertex1->y);
        if (!ClosePoints(&fv, &divends.v2))
        {
        invisseg.type1 = 0;
        invisseg.segend1 = *segend1;
        invisseg.type2 = 1;
        invisseg.v2 = divends.v2;
        addElement(&frontinvissegs, &invisseg);
        invisseg.type2 = 0;
        invisseg.segend2 = *segend1;
        invisseg.type1 = 1;
        invisseg.v1 = divends.v2;
        addElement(&backinvissegs, &invisseg);
        }
    }

childnodes:
    DivideNodeSegs(node->children[0], &frontrealsegs, &frontinvissegs);
    DivideNodeSegs(node->children[1], &backrealsegs, &backinvissegs);

    freeStorage(&frontrealsegs);
    freeStorage(&backrealsegs);
    freeStorage(&frontinvissegs);
    freeStorage(&backinvissegs);
    freeStorage(&cosegends);
    freeStorage(&concursegs);
}

void AddToNodeBBox(int nodenum, storage_t* vertices, fixed_t* bbox)
{
    int             c, i;
    vertex_t*       v;
    storage_t       frontvertices, backvertices;
    node_t*         node;
    unitvector_t    divvector;
    int             side;

    c = getCount(vertices);

    if (bbox)
    {
        for (i=0 ; i<c ; i++)
        {
            v = elementAt(vertices, i);
            M_AddToBox(bbox, v->x, v->y);
        }
    }

    if (nodenum & NF_SUBSECTOR)
        return;

    initStorage(&frontvertices, sizeof(vertex_t));
    initStorage(&backvertices, sizeof(vertex_t));

    node = &nodes[nodenum];
    UnitvectorFromNode(&divvector, node);

    for (i=0 ; i<c ; i++)
    {
        v = elementAt(vertices, i);
        side = VertexOnSide(v, &divvector);
        switch (side)
        {
        case 0:
            addElement(&frontvertices, v);
            break;
        case 1:
            addElement(&backvertices, v);
            break;
        case -1:
            addElement(&frontvertices, v);
            addElement(&backvertices, v);
            break;
        }
    }

    AddToNodeBBox(node->children[0], &frontvertices, node->bbox[0]);
    AddToNodeBBox(node->children[1], &backvertices, node->bbox[1]);

    freeStorage(&frontvertices);
    freeStorage(&backvertices);
}

void MakeSSecPoly(void)
{
    storage_t       realsegs;
    storage_t       invissegs;
    int             index;
    storage_t       vertices;
    int             i, j;
    subsector_t*    sub;
    vertex_t        v;

    initStorage(&realsegs, sizeof(int));
    for (index=0 ; index<numsegs ; index++)
        addElement(&realsegs, &index);

    initStorage(&invissegs, sizeof(invisseg_t));

    DivideNodeSegs(numnodes-1, &realsegs, &invissegs);

    freeStorage(&realsegs);
    freeStorage(&invissegs);

    initStorage(&vertices, sizeof(vertex_t));
    for (i=0 ; i<numsubsectors ; i++)
    {
        sub = &subsectors[i];
        for (j=0 ; j<sub->poly.numvertices ; j++)
        {
            v.x = FLOATTOFIXED(sub->poly.vertices[j].x);
            v.y = FLOATTOFIXED(sub->poly.vertices[j].y);
            addElement(&vertices, &v);
        }
    }

    AddToNodeBBox(numnodes-1, &vertices, NULL);

    freeStorage(&vertices);
}

// @ȉv1.5Œǉ

int buildwithssecpoly = 1;

typedef fvertex_t point_t;

typedef struct
{
	point_t     p1, p2;
	int			linedef;
    int         side;       // 0=right, 1=left side of the above linedef,
    float       offset;
} bspline_t;

typedef	struct
{
	point_t     pt;
	float       dx, dy;
} bspdivline_t;

typedef struct bspstruct_s
{
	storage_t*              lines;      // if non NULL, the node is terminal and has no children
	bspdivline_t            divline;		
	struct	bspstruct_s*    side[2];
} bspnode_t;

#define MAX(a,b) ( (a) > (b) ) ? (a) : (b)
#define MIN(a,b) ( (a) < (b) ) ? (a) : (b)

int Sign(float val)
{
    if (val >= 0)
        return 1;

    return 0;
}

line_t* CheckOverlappingLinedef(line_t* line);

// @one-sided linedef ܂̓ZN^[A܂̓ZN^[O
int InsideSector(point_t p)
{
    line_t* line;
    int     i;
    point_t v1, v2;
    int     sgn1, sgn2, sgn;
    int     ret = 0;

    line = lines;
    for (i=0 ; i<numlines ; i++, line++)
    {
        if (line->sidenum[1] != -1 || line->sidenum[0] == -1)
            continue;

        v1.x = FIXEDTOFLOAT(line->v1->x);
        v1.y = FIXEDTOFLOAT(line->v1->y);
        v2.x = FIXEDTOFLOAT(line->v2->x);
        v2.y = FIXEDTOFLOAT(line->v2->y);

        sgn1 = Sign(v1.y - p.y);
        sgn2 = Sign(v2.y - p.y);
        if (sgn1 != sgn2)
        {
            sgn = Sign(OuterProduct(v2.x - v1.x, v2.y - v1.y, p.x - v1.x, p.y - v1.y));
            if (sgn == sgn2)
            {
                line_t* line_p = CheckOverlappingLinedef(line);
                if (line_p)
                {
                    sgn1 = Sign(FIXEDTOFLOAT(line_p->v1->y) - p.y);
                    sgn2 = Sign(FIXEDTOFLOAT(line_p->v2->y) - p.y);
                    if (sgn1 != sgn2)
                    {
                        I_Debug("InsideSector: crossed one-sided overlapping linedef\n");
                        continue;
                    }
                }
                ret = !ret;
            }
        }
    }

    return ret;
}

bspdivline_t g_divline;

int pointcmp(const void* elem1, const void* elem2)
{
    point_t* p1 = (point_t *)elem1;
    point_t* p2 = (point_t *)elem2;
    float   dp1, dp2;

    dp1 = g_divline.dx*(p1->x - g_divline.pt.x) + g_divline.dy*(p1->y - g_divline.pt.y);
    dp2 = g_divline.dx*(p2->x - g_divline.pt.x) + g_divline.dy*(p2->y - g_divline.pt.y);

    if (dp1 - dp2 > 0)
        return 1;
    if (dp2 - dp1 > 0)
        return -1;
    return 0;
}

storage_t   mapnodestore;
storage_t   mapsubsecstore;
storage_t   mapsegstore;
storage_t   vertexstore;
storage_t   ssecpolystore;

void ProcessSSecPoly(storage_t* store)
{
    polygon_t   poly;
    int         c, i, j, k;
    fvertex_t*  vertices;
    bspline_t*  line;
    fvertex_t   startv, v;

    memset(&poly, 0, sizeof(polygon_t));
    c = store->count;
    if (c < 3)
        I_Debug("ProcessSSecPoly: subsector with less than 3 edges\n");

    vertices = Z_Malloc((c+1)*sizeof(fvertex_t), PU_LEVEL, 0);
    k = 0;
retry:
    i = 0;
    line = elementAt(store, k);
    vertices[i++] = startv = line->p1;
    v = line->p2;
    while (!ClosePoints(&v, &startv) && i < c+1)
    {
        for (j=0 ; j<c ; j++)
        {
            line = elementAt(store, j);
            if (ClosePoints(&v, &line->p1))
            {
                vertices[i++] = v;
                v = line->p2;
                break;
            }
        }
        if (j == c)
        {
            if (i < c && ++k < c)
                goto retry;
            I_Debug("ProcessSSecPoly: subsector not closed\n");
            vertices[i++] = v;
            break;
        }
    }
    poly.numvertices = i;
    poly.vertices = vertices;
    addElement(&ssecpolystore, &poly);
}

int UniqueVertex (float x, float y)
{
    int             i, count;
    /*map*/vertex_t mv, *mvp;
	
	mv.x = FLOATTOFIXED(x);
	mv.y = FLOATTOFIXED(y);
	
// see if an identical vertex already exists
	count = vertexstore.count;
	mvp = elementAt(&vertexstore, 0);
	for (i=0 ; i<count ; i++, mvp++)
		if (mvp->x == mv.x && mvp->y == mv.y)
			return i;

	addElement(&vertexstore, &mv);
	
	return count;	
}

float       bbox[4];

void AddPointToBBox (point_t *pt)
{
	if (pt->x < bbox[BOXLEFT])
		bbox[BOXLEFT] = pt->x;
	if (pt->x > bbox[BOXRIGHT])
		bbox[BOXRIGHT] = pt->x;
		
	if (pt->y > bbox[BOXTOP])
		bbox[BOXTOP] = pt->y;
	if (pt->y < bbox[BOXBOTTOM])
		bbox[BOXBOTTOM] = pt->y;
}

//#define PI  3.14159265359
int numinvissegs;

void ProcessLines (storage_t* store)
{
	int			i,count;
	bspline_t   *wline;
	mapseg_t	line;
	short		angle;
	double		fangle;
	
	bbox[BOXLEFT] = (float)MAXINT;
	bbox[BOXRIGHT] = (float)MININT;
	bbox[BOXTOP] = (float)MININT;
	bbox[BOXBOTTOM] = (float)MAXINT;
	
	count = store->count;
	for (i=0 ; i<count ; i++)
	{
		wline = elementAt(store, i);
		
		memset (&line, 0, sizeof(line));
		AddPointToBBox (&wline->p1);
		AddPointToBBox (&wline->p2);
        if (buildwithssecpoly && wline->side == -1)
        {
            numinvissegs++;
            continue;
        }
		line.v1 = (short)UniqueVertex (wline->p1.x, wline->p1.y);
		line.v2 = (short)UniqueVertex (wline->p2.x, wline->p2.y);
		line.linedef = (short)wline->linedef;
		line.side = (short)wline->side;
		line.offset = (short)wline->offset;
		fangle = atan2 (wline->p2.y - wline->p1.y, wline->p2.x - wline->p1.x);
		angle = (short)(fangle/(PI*2)*0x10000);
		line.angle = angle;
		addElement(&mapsegstore, &line);
	}

    if (buildwithssecpoly)
        ProcessSSecPoly(store);
}

unsigned short ProcessSubsector (storage_t* wmaplinestore)
{
    int				count;
	mapsubsector_t	sub;
	
	memset (&sub,0,sizeof(sub));
	
	count = wmaplinestore->count;
    if (count < 1)
		I_Error ("ProcessSubsector: count = %i",count);

	sub.numsegs = count;
	sub.firstseg = mapsegstore.count;
    if (!buildwithssecpoly)
    ProcessLines (wmaplinestore);
    else
    {
        numinvissegs = 0;
        ProcessLines (wmaplinestore);
        sub.numsegs -= numinvissegs;
    }
    
// add the new subsector
	addElement(&mapsubsecstore, &sub);
	
	return mapsubsecstore.count - 1;
}

unsigned short ProcessNode (bspnode_t *node, short *totalbox)
{
	short		    subbox[2][4];
	int			    i;
    unsigned short  r;
	mapnode_t	    mnode;
	
	memset (&mnode,0,sizeof(mnode));

	if (node->lines)
	{
		r = ProcessSubsector (node->lines);
        //freeStorage(node->lines);
		for (i=0 ; i<4 ; i++)
			totalbox[i] = (short)bbox[i];
		return r | NF_SUBSECTOR;    // NF_SUBSECTOR flags a subsector
	}
	
	mnode.x = (short)node->divline.pt.x;
	mnode.y = (short)node->divline.pt.y;
	mnode.dx = (short)node->divline.dx;
	mnode.dy = (short)node->divline.dy;
	
	r = ProcessNode (node->side[0], subbox[0]);
	mnode.children[0] = r;
	for (i=0 ; i<4 ; i++)
		mnode.bbox[0][i] = subbox[0][i];
	
	r = ProcessNode (node->side[1], subbox[1]);
	mnode.children[1] = r;
	for (i=0 ; i<4 ; i++)
		mnode.bbox[1][i] = subbox[1][i];

	totalbox[BOXLEFT] = MIN(subbox[0][BOXLEFT], subbox[1][BOXLEFT]);
	totalbox[BOXTOP] = MAX(subbox[0][BOXTOP], subbox[1][BOXTOP]);
	totalbox[BOXRIGHT] = MAX(subbox[0][BOXRIGHT], subbox[1][BOXRIGHT]);
	totalbox[BOXBOTTOM] = MIN(subbox[0][BOXBOTTOM], subbox[1][BOXBOTTOM]);
	
	addElement(&mapnodestore, &mnode);
	return mapnodestore.count - 1;
}

void DivlineFromWorldline(bspdivline_t *d, bspline_t *w)
{
	d->pt = w->p1;
	d->dx = w->p2.x - w->p1.x;
	d->dy = w->p2.y - w->p1.y;
}

int PointOnSide(point_t* p, bspdivline_t* l)
{
    float   dx, dy;
    float   px, py;
    float   len;
    float   val;

    len = (float)sqrt(l->dx*l->dx + l->dy*l->dy);
    dx = l->dx / len;
    dy = l->dy / len;

    px = p->x - l->pt.x;
    py = p->y - l->pt.y;

    val = dx*py - px*dy; // 2D cross product

    if (val < -TOLERANCE)
        return 0;
    else if (val > TOLERANCE)
        return 1;
    return -1;
}

int sign (float i)
{
	if (i<-0.1/*0*/) // @2012/02/29
		return -1;
	else if (i>0.1/*0*/)
		return 1;
	return 0;
}

int LineOnSide(bspline_t* wl, bspdivline_t* bl)
{
    int     s1, s2;
    float   dx, dy;
	
	s1 = PointOnSide (&wl->p1, bl);
	s2 = PointOnSide (&wl->p2, bl);

	if (s1 == s2)
	{
		if (s1 == -1)
		{	// colinear, so see if the directions are the same
			dx = wl->p2.x - wl->p1.x;
			dy = wl->p2.y - wl->p1.y;
			if (sign(dx) == sign(bl->dx) && sign(dy) == sign(bl->dy))
				return 0;
			return 1;
		}
		return s1;
	}
	if (s1 == -1)
		return s2;
	if (s2 == -1)
		return s1;

    return -2;
}

float InterceptVector(bspdivline_t* v1, bspdivline_t* v2) // @Real-Time Rendering SE p616 Q
{
    float	frac, num, den;

    den = v1->dx*(-v2->dy) + v1->dy*v2->dx;
    //if (den == 0)
		//I_Error("InterceptVector: parallel");
    num = (v2->pt.x - v1->pt.x)*(-v2->dy) + (v2->pt.y - v1->pt.y)*v2->dx;
    frac = num / den;

    if (frac <= 0.0 || frac >= 1.0)
        I_Debug("InterceptVector: intersection outside line\n");

    return frac;
}

bspline_t* CutLine(bspline_t *wl, bspdivline_t *bl)
{
    int         side;
    bspline_t*  new_p;
    bspdivline_t wld;
    float       frac;
    point_t     intr;
    float       offset;

    DivlineFromWorldline(&wld, wl);
    //new_p = malloc(sizeof(bspline_t));
    new_p = Z_Malloc(sizeof(bspline_t), PU_LEVEL, 0);
    //memset(new_p,0,sizeof(*new_p));
    *new_p = *wl;

    frac = InterceptVector(&wld, bl);
	intr.x = wld.pt.x + wld.dx*frac;
	intr.y = wld.pt.y + wld.dy*frac;
	offset = wl->offset + frac*(float)sqrt(wld.dx*wld.dx+wld.dy*wld.dy);
	side = PointOnSide(&wl->p1, bl);
	if (side == 0)
	{	// line starts on front side
		wl->p2 = intr;
		new_p->p1 = intr;
		new_p->offset = offset;
	}
	else
	{	// line starts on back side
		wl->p1 = intr;
		wl->offset = offset;
		new_p->p2 = intr;
	}

    return new_p;
}

void ExecuteSplit(storage_t* bsplines, bspline_t* spliton, storage_t* frontlist, storage_t* backlist)
{
    int             i,c,side;
    bspline_t       *line_p, *newline_p;
    bspdivline_t    divline;

    DivlineFromWorldline (&divline, spliton);
	
	c = bsplines->count;
		
	for (i=0 ; i<c ; i++)
	{
		line_p = elementAt(bsplines, i);
		if (line_p == spliton)
			side = 0;
		else
			side = LineOnSide (line_p, &divline);
		switch (side)
		{
		case 0:
			addElement(frontlist, line_p);
			break;
		case 1:
			addElement(backlist, line_p);
			break;
		case -2:
			newline_p = CutLine (line_p, &divline);
			addElement(frontlist, line_p);
			addElement(backlist, newline_p);
			break;
		//default:
			//Error ("ExecuteSplit: bad side");
		}
	}

    if (buildwithssecpoly)
    {
        storage_t   colipoints;
        int         j;
        point_t     *p1, *p2;
        bspline_t   line;

        initStorage(&colipoints, sizeof(point_t));

        c = frontlist->count;
        for (i=0 ; i<c ; i++)
        {
            line_p = elementAt(frontlist, i);
            side = PointOnSide(&line_p->p1, &divline);
            if (side == -1)
                addElement(&colipoints, &line_p->p1);
            side = PointOnSide(&line_p->p2, &divline);
            if (side == -1)
                addElement(&colipoints, &line_p->p2);
        }

        c = backlist->count;
        for (i=0 ; i<c ; i++)
        {
            line_p = elementAt(backlist, i);
            side = PointOnSide(&line_p->p1, &divline);
            if (side == -1)
                addElement(&colipoints, &line_p->p1);
            side = PointOnSide(&line_p->p2, &divline);
            if (side == -1)
                addElement(&colipoints, &line_p->p2);
        }

        g_divline = divline;
        c = colipoints.count;
        qsort(elementAt(&colipoints, 0), c, sizeof(point_t), pointcmp);

        p1 = elementAt(&colipoints, 0);
        for (i=1 ; i<c ; i++)
        {
            p2 = elementAt(&colipoints, i);

            if (ClosePoints(p1, p2))
            {
                p1 = p2;
                continue;
            }

            for (j=0 ; j<bsplines->count ; j++)
            {
                line_p = elementAt(bsplines, j);
                if ((ClosePoints(&line_p->p1, p1) && ClosePoints(&line_p->p2, p2)) ||
                    (ClosePoints(&line_p->p2, p1) && ClosePoints(&line_p->p1, p2)))
                    break;
            }
            if (j != bsplines->count)
            {
                p1 = p2;
                continue;
            }

            if (1)
            {
                point_t     midpoint;

                midpoint.x = p1->x + 0.5f*(p2->x - p1->x);
                midpoint.y = p1->y + 0.5f*(p2->y - p1->y);
                if (!InsideSector(midpoint))
                {
                    p1 = p2;
                    continue;
                }
            }
            else
            {
                storage_t   concurlines;
                int*        check;
                int         k;
                bspline_t   *line1, *line2;
                int         backside = 0;

                initStorage(&concurlines, sizeof(bspline_t));
                for (j=0 ; j<frontlist->count ; j++)
                {
                    line_p = elementAt(frontlist, j);
                    if (ClosePoints(&line_p->p1, p1) || ClosePoints(&line_p->p2, p1))
                        addElement(&concurlines, line_p);
                }
                for (j=0 ; j<backlist->count ; j++)
                {
                    line_p = elementAt(backlist, j);
                    if (ClosePoints(&line_p->p1, p1) || ClosePoints(&line_p->p2, p1))
                        addElement(&concurlines, line_p);
                }
                check = (int *)alloca(concurlines.count*sizeof(int));
                memset(check, 0, concurlines.count*sizeof(int));
                for (j=0 ; j<concurlines.count ; j++)
                {
                    for (k=j+1 ; k<concurlines.count ; k++)
                    {
                        line1 = elementAt(&concurlines, j);
                        line2 = elementAt(&concurlines, k);
                        if (ClosePoints(&line1->p1, &line2->p2) && ClosePoints(&line1->p2, &line2->p1))
                            check[j] = check[k] = 1;
                    }
                }
                for (j=0 ; j<concurlines.count ; j++)
                {
                    if (check[j])
                        continue;
                    line_p = elementAt(&concurlines, j);
                    DivlineFromWorldline(&divline, line_p);
                    side = PointOnSide(p2, &divline);
                    if (side == 1 || side == -1)
                        backside++;
                }
                if (backside >= 2)
                {
                    p1 = p2;
                    continue;
                }
            }

            line.side = -1;
            line.offset = 0; // @CutLine()ŎĝňꉞKȒlĂ
            line.p1 = *p1;
            line.p2 = *p2;
            addElement(frontlist, &line);
            line.p1 = *p2;
            line.p2 = *p1;
            addElement(backlist, &line);

            p1 = p2;
        }

        freeStorage(&colipoints);
    }
}

int EvaluateSplit(storage_t* bsplines, bspline_t* spliton, int bestgrade)
{
    int             i,c,side;
	bspline_t*      line_p;
	bspdivline_t    divline;
	int             frontcount, backcount, max, inc;
	int             grade;

    DivlineFromWorldline(&divline, spliton);

    frontcount = backcount = 0;
	c = bsplines->count;
	grade = 0;

    for (i=0 ; i<c ; i++)
	{
		line_p = elementAt(bsplines, i);
        if (line_p->side == -1) // @invisible line  convexicity test ̑Ώۂɂ͓Ȃ͗l
            continue;
		if (line_p == spliton)
			side = 0;
		else
			side = LineOnSide(line_p, &divline);
		switch (side)
		{
		case 0:
			frontcount++;
			break;
		case 1:
			backcount++;
			break;
		case -2:
			frontcount++;
			backcount++;
			break;
		}
		
		max = MAX(frontcount,backcount);
		inc = (frontcount+backcount) - c;
		grade = max+inc*8;
		if (grade > bestgrade)
			return grade;		// might as well stop now
	}

    if (frontcount == 0 || backcount == 0)
		return MAXINT;			// line does not partition at all

    return grade;
}

bspnode_t* BSPList(storage_t* bsplines)
{
    bspnode_t*  node_p;
    int         i, c, step;
    int         v, bestv;
    bspline_t   *line_p, *bestline_p;
    storage_t   *frontlist, *backlist;

    //node_p = (bspnode_t *)malloc(sizeof(*node_p));
    node_p = Z_Malloc(sizeof(*node_p), PU_LEVEL, 0);
	memset(node_p, 0, sizeof(*node_p));

    c = bsplines->count;
	bestv = MAXINT;	
	bestline_p = NULL;
	step = (c/40)+1;    // set this to 1 for an exhaustive search
research:
	for (i=0 ; i<c ; i+=step)
	{
		line_p = elementAt(bsplines, i);
        if (line_p->side == -1)
            continue;
		v = EvaluateSplit(bsplines, line_p, bestv);
		if (v < bestv)
		{
			bestv = v;
			bestline_p = line_p;
		}
	}

//
// if none of the lines should be split, the remaining lines
// are convex, and form a terminal node
//
	if (bestv == MAXINT)
	{
		if (step > 1)
		{
			step = 1;
			goto research;
		}
        node_p->lines = bsplines;
		return node_p;
	}

    DivlineFromWorldline(&node_p->divline, bestline_p);

    //frontlist = malloc(sizeof(storage_t));
    frontlist = Z_Malloc(sizeof(storage_t), PU_LEVEL, 0);
    //backlist = malloc(sizeof(storage_t));
    backlist = Z_Malloc(sizeof(storage_t), PU_LEVEL, 0);
    initStorage(frontlist, sizeof(bspline_t));
    initStorage(backlist, sizeof(bspline_t));

    ExecuteSplit(bsplines, bestline_p, frontlist, backlist);

	node_p->side[0] = BSPList(frontlist);
	node_p->side[1] = BSPList(backlist);

    freeStorage(bsplines);
    //free(frontlist);
    //free(backlist);

    return node_p;
}

line_t* CheckOverlappingLinedef(line_t* line)
{
    bspdivline_t    divline;
    point_t         pt;
    line_t*         line_p;
    int             s1, s2;
    float           x1, y1, x2, y2;

    pt.x = FIXEDTOFLOAT(line->v1->x);
    pt.y = FIXEDTOFLOAT(line->v1->y);
    divline.pt = pt;
    pt.x = FIXEDTOFLOAT(line->v2->x);
    pt.y = FIXEDTOFLOAT(line->v2->y);
    divline.dx = pt.x - divline.pt.x;
    divline.dy = pt.y - divline.pt.y;

    line_p = lines;
    while (line_p != line)
    {
        if (line->v1 == line_p->v1 && line->v2 == line_p->v2)
        {
            I_Debug("CheckOverlappingLinedef: found  an identical linedef\n");
            return line_p;
        }

        pt.x = x1 = FIXEDTOFLOAT(line_p->v1->x);
        pt.y = y1 = FIXEDTOFLOAT(line_p->v1->y);
        s1 = PointOnSide(&pt, &divline);
        pt.x = x2 = FIXEDTOFLOAT(line_p->v2->x);
        pt.y = y2 = FIXEDTOFLOAT(line_p->v2->y);
        s2 = PointOnSide(&pt, &divline);
        if (s1 == -1 && s2 == -1)
        {
            // colinear, so see if the directions are the same
			float dx = x2 - x1;
			float dy = y2 - y1;
            if (sign(dx) == sign(divline.dx) && sign(dy) == sign(divline.dy))
            {
                if (line->v2 != line_p->v1 && line_p->v2 != line->v1)
                {
                    if (divline.dx > 1.0f)
                    {
                        if (x1 < (divline.pt.x + divline.dx) && x2 > divline.pt.x)
                            return line_p;
                    }
                    else if (divline.dx < -1.0f)
                    {
                        if (x1 > (divline.pt.x + divline.dx) && x2 < divline.pt.x)
                            return line_p;
                    }
                    else
                    {
                        if (divline.dy > 0)
                        {
                            if (y1 < (divline.pt.y + divline.dy) && y2 > divline.pt.y)
                                return line_p;
                        }
                        else
                        {
                            if (y1 > (divline.pt.y + divline.dy) && y2 < divline.pt.y)
                                return line_p;
                        }
                    }
                }
            }
        }
        line_p++;
    }

    return NULL;
}

void RebuildBSP(void)
{
    storage_t   bsplines;
    line_t*     line;
    int         i,j,k;
    point_t     pt;
    bspline_t   li;
    bspnode_t*  startnode;
    vertex_t*   vp;
    short       worldbounds[4];
    mapsubsector_t*	ms;
    subsector_t*	ss;
    mapnode_t*	mn;
    node_t*	no;
    mapseg_t*		ml;
    seg_t*		se;
    line_t*		ldef;
    int			linedef;
    int			side;
    float       dx, dy;

    initStorage(&bsplines, sizeof(bspline_t));
    line = lines;
    for (i=0 ; i<numlines ; i++, line++)
    {
        // @trip wireƂĂ݂̂linedefł\l
        if (line->sidenum[0] == -1 && line->sidenum[1] == -1)
            continue;

        if (CheckOverlappingLinedef(line))
        {
            I_Debug("RebuildBSP: found  an overlapping linedef %d\n", i);
            if (line->sidenum[1] == -1)
                I_Debug("\tOne-Sided\n");
        }

        pt.x = FIXEDTOFLOAT(line->v1->x);
        pt.y = FIXEDTOFLOAT(line->v1->y);
        li.p1 = pt;
        pt.x = FIXEDTOFLOAT(line->v2->x);
        pt.y = FIXEDTOFLOAT(line->v2->y);
        li.p2 = pt;
        li.linedef = i;
        li.side = 0;
        li.offset = 0;
        addElement(&bsplines, &li);

        if (line->sidenum[1] != -1)
        {
            pt.x = FIXEDTOFLOAT(line->v2->x);
            pt.y = FIXEDTOFLOAT(line->v2->y);
            li.p1 = pt;
            pt.x = FIXEDTOFLOAT(line->v1->x);
            pt.y = FIXEDTOFLOAT(line->v1->y);
            li.p2 = pt;
            li.linedef = i;
            li.side = 1;
            li.offset = 0;
            addElement(&bsplines, &li);
        }
    }

    startnode = BSPList(&bsplines);

    initStorage(&mapnodestore, sizeof(mapnode_t));
    initStorage(&mapsubsecstore, sizeof(mapsubsector_t));
    initStorage(&mapsegstore, sizeof(mapseg_t));
    initStorage(&vertexstore, sizeof(vertex_t));
    initStorage(&ssecpolystore, sizeof(polygon_t));

    vp = vertexes;
    for (i=0 ; i<numvertexes ; i++, vp++)
        addElement(&vertexstore, vp);

    ProcessNode (startnode, worldbounds);

    numvertexes = vertexstore.count;
    vertexes = Z_Malloc (numvertexes*sizeof(vertex_t),PU_LEVEL,0);
    memcpy(vertexes, vertexstore.data, numvertexes*sizeof(vertex_t));

    numsubsectors = mapsubsecstore.count;
    subsectors = Z_Malloc (numsubsectors*sizeof(subsector_t),PU_LEVEL,0);
    ms = (mapsubsector_t *)mapsubsecstore.data;
    memset (subsectors,0, numsubsectors*sizeof(subsector_t));
    ss = subsectors;
    for (i=0 ; i<numsubsectors ; i++, ss++, ms++)
    {
	ss->numlines = ms->numsegs;
	ss->firstline = ms->firstseg;
    if (buildwithssecpoly)
    {
        polygon_t*  poly = elementAt(&ssecpolystore, i);
        ss->poly = *poly;
    }
    }

    numnodes = mapnodestore.count;
    nodes = Z_Malloc (numnodes*sizeof(node_t),PU_LEVEL,0);
    mn = (mapnode_t *)mapnodestore.data;
    no = nodes;
    for (i=0 ; i<numnodes ; i++, no++, mn++)
    {
	no->x = mn->x<<FRACBITS;
	no->y = mn->y<<FRACBITS;
	no->dx = mn->dx<<FRACBITS;
	no->dy = mn->dy<<FRACBITS;
	for (j=0 ; j<2 ; j++)
	{
	    no->children[j] = mn->children[j];
	    for (k=0 ; k<4 ; k++)
		no->bbox[j][k] = mn->bbox[j][k]<<FRACBITS;
	}
    }

    numsegs = mapsegstore.count;
    segs = Z_Malloc (numsegs*sizeof(seg_t),PU_LEVEL,0);	
    memset (segs, 0, numsegs*sizeof(seg_t));
    ml = (mapseg_t *)mapsegstore.data;
    se = segs;
    for (i=0 ; i<numsegs ; i++, se++, ml++)
    {
	se->v1 = &vertexes[ml->v1];
	se->v2 = &vertexes[ml->v2];
	se->angle = (ml->angle)<<16;
	se->offset = (ml->offset)<<16;
	linedef = ml->linedef;
	ldef = &lines[linedef];
	se->linedef = ldef;
	side = ml->side;
	se->sidedef = &sides[ldef->sidenum[side]];
	se->frontsector = sides[ldef->sidenum[side]].sector;
	if (ldef-> flags & ML_TWOSIDED)
	    se->backsector = sides[ldef->sidenum[side^1]].sector;
	else
	    se->backsector = 0;

    dx = FIXEDTOFLOAT(se->v2->x - se->v1->x);
    dy = FIXEDTOFLOAT(se->v2->y - se->v1->y);
    se->length = (float)sqrt(dx*dx + dy*dy);
    }

    freeStorage(&mapnodestore);
    freeStorage(&mapsubsecstore);
    freeStorage(&mapsegstore);
    freeStorage(&vertexstore);
    freeStorage(&ssecpolystore);
}
